import { ErrorRequestHandler, Request, RequestHandler, Router } from 'express';
import config from 'yandex-cfg';
import proxy from 'express-http-proxy';
import index from 'server/controllers/report';
import dev from 'server/controllers/dev';
import admin from 'server/controllers/admin';
import exam from 'server/controllers/exam';
import videos from 'server/controllers/videos';
import nullify from 'server/controllers/nullify';
import ban from 'server/controllers/ban';

import {
    mockBlackbox,
    blackbox,
    cookieParser,
    helmet,
    httpGeobase,
    httpLangdetect,
    httpUatraits,
    renewBbSession,
    secretkey,
    tld,
    tvm,
    xForwardedForFix,
    yandexCsp,
    yandexuid,
    base,
    access
} from 'server/middlewares';

import { serverUrls } from 'common/server-urls';
import { apiUrls } from 'common/api-urls';

const middlewares: Array<RequestHandler | ErrorRequestHandler> = [
    cookieParser,
    tld,
    tvm,
    mockBlackbox,
    blackbox, // Depends on cookieParser and tvm
    renewBbSession, // Depends on blackbox and tld
    helmet,
    yandexCsp, // Depends on cookieParser
    xForwardedForFix,
    httpGeobase,
    httpLangdetect, // Depends on cookieParser and httpGeobase
    httpUatraits,
    yandexuid, // Depends on cookieParser and httpUatraits
    secretkey, // Depends on blackbox and yandexuid
    base
];

/* eslint-disable max-statements */
export default function (router: Router) {
    router.post(serverUrls.freeze.template, proxy(config.api.host, {
        proxyReqPathResolver: () => {
            return apiUrls.freeze.template;
        }
    }));
    router.post(serverUrls.unfreeze.template, proxy(config.api.host, {
        proxyReqPathResolver: () => {
            return apiUrls.unfreeze.template;
        }
    }));
    router.post(serverUrls.syncAgenciesInfo.template, proxy(config.api.host, {
        proxyReqPathResolver: () => {
            return apiUrls.syncAgenciesInfo.template;
        }
    }));
    router.post(serverUrls.upload.template, proxy(config.api.host, {
        proxyReqPathResolver: () => {
            return apiUrls.parseXLSX.template;
        }
    }));
    router.post(serverUrls.publishExam.template, proxy(config.api.host, {
        proxyReqPathResolver: () => {
            return apiUrls.loadData.template;
        }
    }));
    router.get(serverUrls.attemptVideos.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            const { attemptId } = req.params;

            return `${apiUrls.videos.template}/${attemptId}`;
        }
    }));
    router.post(serverUrls.appeal.template, proxy(config.api.host, {
        proxyReqPathResolver: () => {
            return apiUrls.appeal.template;
        }
    }));
    router.post(serverUrls.examSettings.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.examSettings.build(req.params);
        }
    }));
    router.get(serverUrls.downloadJSON.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.downloadJSON.build(req.params);
        }
    }));
    router.get(serverUrls.report.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.report.build(req.params, req.query);
        }
    }));
    router.get(serverUrls.getAssociationsInfo.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.getAssociationsInfo.build({}, req.query);
        }
    }));
    router.post(serverUrls.getAttemptsInfo.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.getAttemptsInfo.build(req.params);
        }
    }));
    router.post(serverUrls.getCertificatesInfo.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.getCertificatesInfo.build(req.params);
        }
    }));
    router.post(serverUrls.nullifyAttempt.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.nullifyAttempt.build(req.params);
        }
    }));
    router.post(serverUrls.nullifyCertificates.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.nullifyCertificates.build(req.params);
        }
    }));
    router.post(serverUrls.associateUsers.template, proxy(config.api.host, {
        proxyReqPathResolver: () => {
            return apiUrls.associateUsers.template;
        }
    }));
    router.post(serverUrls.ban.template, proxy(config.api.host, {
        proxyReqPathResolver: () => {
            return apiUrls.ban.template;
        }
    }));
    router.post(serverUrls.unban.template, proxy(config.api.host, {
        proxyReqPathResolver: () => {
            return apiUrls.unban.template;
        }
    }));
    router.post(serverUrls.lockExam.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.lock.build(req.params);
        }
    }));
    router.post(serverUrls.unlockExam.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.unlock.build(req.params);
        }
    }));
    router.get(serverUrls.lastVersion.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.lastVersion.build(req.params);
        }
    }));
    router.post(serverUrls.draft.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.draft.build(req.params);
        }
    }));
    router.delete(serverUrls.draft.template, proxy(config.api.host, {
        proxyReqPathResolver: (req: Request) => {
            return apiUrls.draft.build(req.params);
        }
    }));
    router.post(serverUrls.draftModeration.template, proxy(config.api.host, {
        proxyReqPathResolver: () => {
            return apiUrls.draftModeration.build();
        }
    }));

    router.get(serverUrls.root.template, middlewares, access('reports'), index);
    router.get(serverUrls.videos.template, middlewares, access('videos'), videos);
    router.get(serverUrls.dev.template, middlewares, access('dev'), dev);
    router.get(serverUrls.admin.template, middlewares, access('admin'), admin);
    router.get(serverUrls.exam.template, middlewares, access('exam'), exam);
    router.get(serverUrls.nullify.template, middlewares, access('nullify'), nullify);
    router.get(serverUrls.banPage.template, middlewares, access('bans'), ban);
}
/* eslint-enable max-statements */
