package goblogs

import (
	"fmt"
	"net/http"

	"github.com/go-resty/resty/v2"
)

const (
	serviceTicketHeader = "x-ya-service-ticket"
)

type Client struct {
	*resty.Client
}

type Options struct {
	BaseURL       string
	ServiceTicket string
	RetryCount    int

	CustomHeaders map[string]string
}

func New(options Options) *Client {
	restyClient := resty.
		New().
		SetRetryCount(options.RetryCount).
		SetHostURL(options.BaseURL).
		SetHeaders(options.CustomHeaders).
		SetHeader(serviceTicketHeader, options.ServiceTicket)

	return &Client{restyClient}
}

type APIError struct {
	Code    string `json:"internalCode"`
	Message string `json:"message"`
}

func (err *APIError) Error() string {
	return fmt.Sprintf("%s: %s", err.Code, err.Message)
}

type BlogIdentity struct {
	client *Client

	Slug     string
	Language string
}

func (c *Client) BlogIdentity(slug, language string) *BlogIdentity {
	return &BlogIdentity{c, slug, language}
}

func (b *BlogIdentity) getJSONGeneric(urlPattern string, result interface{}, r *resty.Request) (interface{}, error) {
	response, err := r.
		SetResult(result).
		SetError(APIError{}).
		SetPathParams(map[string]string{
			"blog": b.Slug,
		}).
		SetQueryParam("lang", b.Language).
		Get(urlPattern)

	if err != nil {
		return nil, err
	}

	if statusCode := response.StatusCode(); statusCode != http.StatusOK {
		err := response.Error().(error)

		if err.(*APIError).Code == "" {
			err = fmt.Errorf(`failure, blog: "%s" language: "%s" status: %d`, b.Slug, b.Language, statusCode)
		}

		return nil, err
	}

	return response.Result(), nil
}

func (b *BlogIdentity) getJSON(urlPattern string, result interface{}) (interface{}, error) {
	return b.getJSONGeneric(urlPattern, result, b.client.R())
}

func (b *BlogIdentity) GetInfo() (*Blog, error) {
	result, err := b.getJSON("/{blog}", Blog{})

	if err != nil {
		return nil, err
	}

	return result.(*Blog), nil
}

func (b *BlogIdentity) GetAllCategories() ([]Category, error) {
	result, err := b.getJSON("/categories/all/{blog}", []Category{})

	if err != nil {
		return nil, err
	}

	return *result.(*[]Category), nil
}

func (b *BlogIdentity) GetAllTags() ([]Tag, error) {
	result, err := b.getJSON("/tags/all/{blog}", []Tag{})

	if err != nil {
		return nil, err
	}

	return *result.(*[]Tag), nil
}

func (b *BlogIdentity) GetAllPosts() ([]Post, error) {
	var result []Post
	var fromPost string

	hasNext := true

	for hasNext {
		baseRequest := b.client.R().SetQueryParam("size", "100")

		if fromPost != "" {
			baseRequest = baseRequest.SetQueryParam("from", fromPost)
		}

		response, err := b.getJSONGeneric("/posts/{blog}", []Post{}, baseRequest)

		if err != nil {
			return nil, err
		}

		posts := *response.(*[]Post)

		result = append(result, posts...)

		if ln := len(posts); ln != 0 {
			fromPost = posts[ln-1].ID
			hasNext = posts[ln-1].HasNext
		} else {
			hasNext = false
		}
	}

	return result, nil
}

func (b *BlogIdentity) GetArchivePostsByYear(year uint) ([]Post, error) {
	r := b.client.R().SetQueryParam("year", fmt.Sprint(year))
	result, err := b.getJSONGeneric("/posts/archive/{blog}", []Post{}, r)

	if err != nil {
		return nil, err
	}

	return *result.(*[]Post), nil
}

type PostIdentity struct {
	blogIdentity *BlogIdentity

	Slug string
}

func (b *BlogIdentity) PostIdentity(slug string) *PostIdentity {
	return &PostIdentity{b, slug}
}

func (p *PostIdentity) getJSONGeneric(urlPattern string, result interface{}, r *resty.Request) (interface{}, error) {
	response, err := r.
		SetResult(result).
		SetError(APIError{}).
		SetPathParams(map[string]string{
			"blog": p.blogIdentity.Slug,
			"post": p.Slug,
		}).
		SetQueryParam("lang", p.blogIdentity.Language).
		Get(urlPattern)

	if err != nil {
		return nil, err
	}

	if statusCode := response.StatusCode(); statusCode != http.StatusOK {
		err := response.Error().(error)

		if err.(*APIError).Code == "" {
			err = fmt.Errorf(`failure, blog: "%s" language: "%s" post: "%s" status: %d`, p.blogIdentity.Slug, p.blogIdentity.Language, p.Slug, statusCode)
		}

		return nil, err
	}

	return response.Result(), nil
}

func (p *PostIdentity) getJSON(urlPattern string, result interface{}) (interface{}, error) {
	return p.getJSONGeneric(urlPattern, result, p.blogIdentity.client.R())
}

func (p *PostIdentity) GetInfo() (*Post, error) {
	result, err := p.getJSON("/post/{blog}/{post}", Post{})

	if err != nil {
		return nil, err
	}

	return result.(*Post), nil
}

func (p *PostIdentity) GetAllComments() ([]Comment, error) {
	result, err := p.getJSON("/comments/all/{blog}/{post}", []Comment{})

	if err != nil {
		return nil, err
	}

	return *result.(*[]Comment), nil
}

func (p *PostIdentity) GetAllRelatedArticles() ([]RelatedArticle, error) {
	result, err := p.getJSON("/post/related/{blog}/{post}", []RelatedArticle{})

	if apiErr, ok := err.(*APIError); ok && apiErr.Code == "400_PNR" {
		return []RelatedArticle{}, nil
	}

	if err != nil {
		return nil, err
	}

	return *result.(*[]RelatedArticle), nil
}
