'use strict';

function assertWarn(expr, message) {
    if (!expr) {
        console.warn(message);
    }
}

const { formats } = require('@yandex-int/palmsync');

const {
    palmsync_s3MdsAccessKeyId,
    palmsync_s3MdsAccessSecretKey,
    palmsync_testpalmToken,
    STARTREK_TOKEN
} = process.env;

assertWarn(palmsync_testpalmToken,
    'Define palmsync_testpalmToken env variable: https://nda.ya.ru/3UX83d');
assertWarn(palmsync_s3MdsAccessKeyId,
    'Define palmsync_s3MdsAccessKeyId env variable: https://wiki.yandex-team.ru/mds/s3-api/authorization/');
assertWarn(palmsync_s3MdsAccessSecretKey,
    'Define palmsync_s3MdsAccessSecretKey env variable: https://wiki.yandex-team.ru/mds/s3-api/authorization/');
assertWarn(STARTREK_TOKEN,
    'Define STARTREK_TOKEN env variable: https://nda.ya.ru/3TBofc');

module.exports = {
    project: 'connect',
    schemeExtension: [
        {
            // Для сравнения тестов с одинаковыми тайтлами. Заполняется автоматически. Служебное.
            name: 'titleSuffix',
            required: false,
            meta: true
        },
        {
            name: 'priority',
            required: true,
            meta: true,
            format: formats.isEnum('blocker', 'critical', 'normal', 'minor')
        },
        {
            name: 'tlds',
            required: true,
            meta: true
        }
    ],
    s3MdsUpload: true,
    s3MdsBucketName: 'crowdtest',
    concurrency: 1,
    retries: 3,
    timeout: 10000,
    validationOpts: {
        mode: ['steps-group-syntax', 'automation-syntax']
    },
    synchronizationOpts: {
        supportStepsGroup: true,
        // https://github.yandex-team.ru/search-interfaces/infratest/blob/master/packages/palmsync/CHANGELOG.md#850-2019-10-10
        maxSoftErrorsThreshold: 3
    },
    hermioneReportPath: '.tests//hermione/report/hermione-report.json',
    hermioneConfigPath: 'tests/hermione/config.js',
    sets: {
        yaml: {
            desktop: {
                specs: [
                    'tests/hermione/suites/**/*.testpalm.yml'
                ],
                envs: [{
                    PLATFORM: 'desktop',
                    CONNECT_HOST:'https://connect-assessors.ws.yandex.ru',
                    TIMESTAMP: Date.now()
                }],
                baseUrl: '/',
                browsers: ['chrome']
            }
        }
    },
    stepsGroupPaths: [
        'tests/hermione/**/*.testpalm.steps.yml'
    ],
    testCaseDecorator(test) {
        const titleSuffix = '({{PLATFORM}}{{DELIMITER}}{{BROWSER_TYPE}})';

        test.title += ` ${titleSuffix}`;
        test.setField('platform', '{{PLATFORM}}');
        test.setField('titleSuffix', titleSuffix);
    },
    fullTitleComparator(testCase, hermioneTestTitle) {
        return testCase.fullTitle.toLowerCase() ===
            `${hermioneTestTitle.toLowerCase()} ${testCase.getField('titleSuffix')}`;
    },
    plugins: {}
};

// Костыль! Дефолтное значение из cli опций перетирает опцию из конфига
// Настройка переменными окружения более приоритетна, поэтому перебиваю
process.env.palmsync_s3MdsUpload = module.exports.s3MdsUpload;
