import _ from 'lodash';
import { ActionTypes } from 'constants/Common';
import dispatcher from 'dispatcher';

import Alert from 'components/ModalForms/Alert';
import Confirm from 'components/ModalForms/Confirm';
import Intro from 'components/Intro';
import SettingsStore from 'stores/Settings';
import SettingsActions from 'actions/Settings';

export default {
    clear,
    close,
    open,
    alert,
    confirm,
    openIntro,
};

/**
 * Открывает диалог с переданными данными
 * @method open
 * @param  {Object}    data
 * @param  {String}    [data.title]     Заголовок модального окна
 * @param  {Boolean}   [data.closable]  Поддержка событий с клавиатуры и закрытие по крестику
 * @param  {function}  data.component   React компонент который будет отрисован
 * @param  {Object}    [data.props]     Параметры с которыми будет отрисован переданный компонент
 */
function open(data) {
    dispatcher.dispatch(ActionTypes.OPEN_MODAL_DIALOG, data);
}

/**
 * Закрывает диалог
 * @method close
 */
function close() {
    dispatcher.dispatch(ActionTypes.CLOSE_MODAL_DIALOG);
}

/**
 * Закрывает диалог и очищает от старых данных
 * @method clear
 */
function clear() {
    dispatcher.dispatch(ActionTypes.CLEAR_MODAL_DIALOG);
}

function alert(options) {
    open({
        mod: 'alert',
        title: options.title,
        component: Alert,
        props: _.omit(options, 'title'),
    });
}

function confirm(options) {
    open({
        closable: options.hasOwnProperty('closable') ? options.closable : true,
        mod: 'confirm',
        title: options.title,
        component: Confirm,
        props: _.omit(options, 'title', 'onClose'),
        onClose: options.onClose,
    });
}

function openIntro(options) {
    let onClose;

    if (options.appearsOnce) {
        const id = (options.id || options.mod || '').replace(/\-/g, '_');
        const key = id ? `${id}_visited` : null;

        if (key && SettingsStore.get(key)) {
            return;
        }

        onClose = () => {
            SettingsActions.updateSettings({
                [key]: true,
            });
        };
    }

    open({
        component: Intro,
        mod: 'intro',
        props: _.omit(options, 'appearsOnce'),
        onClose,
    });
}
