import getErrors from 'lib/getErrors';

import ModalActions from 'actions/Modal';
import SettingsActions from 'actions/Settings';

import NewsMailSubscription from 'components/NewsMailSubscription';

import ConfigStore from 'stores/Config';
import SettingsStore from 'stores/Settings';
import apiSender from '../api/sender';

export default {
    showSubscriptionForm,
    subscribeEmail,
    requestEmail,
};

const DAY_LENGTH = 86400000;
// показываем окно обеления рассылок через 3 дня после создания организации
const DAYS_BEFORE_SHOW_NEWS_LIST = 3;
// показываем окно обеления рассылок через 7 дней после последнего показа
const DAYS_AFTER_LAST_SHOWN = 7;

/**
 * Добавление подтвержденной почты в список рассылок
 * @method subscribeEmail
 * @param  {Object} options
 * @returns {Promise}
 */
function subscribeEmail(options) {
    return apiSender.subscribe(options)
        .then(response => {
            const errors = getErrors(response);

            if (!errors.isEmpty()) {
                return { errors };
            }

            return response;
        });
}

/**
 * Отправка письма пользователю (на указанную им почту) для подписки по схеме double opt-in
 * @method requestEmail
 * @param  {Object} options
 * @returns {Promise}
 */
function requestEmail(options) {
    return apiSender.request(options)
        .then(response => {
            const errors = getErrors(response);

            if (!errors.isEmpty()) {
                return { errors };
            }

            return response;
        })
        .catch(error => ({ errors: getErrors(error) }));
}

function showSubscriptionForm() {
    const userSettings = SettingsStore.getUserSettings() || window.ya.connect.initial.user_settings;
    const subscriptionData = userSettings.news_mail_subscription;
    const firstVisit = userSettings.first_visit || subscriptionData.first_visit;

    if (_shouldShowSubscriptionForm(subscriptionData, firstVisit)) {
        const updateData = {
            news_mail_subscription: {
                last_shown: Date.now(),
            },
        };

        // проставляем время последнего показа
        SettingsActions.updateSettings(updateData);

        return ModalActions.open({
            component: NewsMailSubscription,
            props: {
                close: () => ModalActions.close(),
            },
        });
    }

    // проставляем время первого визита, если оно не проставлено и это не дефолтные настройки
    if (!firstVisit && !userSettings.default) {
        return SettingsActions.updateSettings({ first_visit: Date.now() });
    }
}

function _shouldShowSubscriptionForm({ declined, confirm, last_shown: lastShown }, firstVisit) {
    if (!ConfigStore.get('ui.showSubscriptionForm')) {
        return false;
    }

    // Не показываем при первом посещении
    if (!firstVisit) {
        return false;
    }

    // Пользователь отказался от подписки
    if (declined) {
        return false;
    }

    // Пользователь подписан
    if (confirm) {
        return false;
    }

    // С момента первого посещения прошло меньше $(DAYS_BEFORE_SHOW_NEWS_LIST) дней
    if (Date.now() - firstVisit < DAYS_BEFORE_SHOW_NEWS_LIST * DAY_LENGTH) {
        return false;
    }

    // Показываем первый раз
    if (!lastShown) {
        return true;
    }

    // Показываем если с момента последнего показа прошло $(DAYS_AFTER_LAST_SHOWN) дней
    return Date.now() - lastShown > DAYS_AFTER_LAST_SHOWN * DAY_LENGTH;
}
