import PropTypes from 'prop-types';
import React from 'react';
import Menu from 'react-bem-components/lib/Menu';
import { Permissions } from 'constants/Common';
import PermissionStore from 'stores/Permissions';
import DepartmentStore from 'stores/Departments';
import { i18n } from 'lib/i18n';
import Metrika from 'lib/metrika';
import timer from 'lib/timer';

import ModalActions from 'actions/Modal';
import UserModalActions from 'actions/Modal.User';
import InviteModalActions from 'actions/Modal.Invite';

import DepartmentForm from 'components/Department/Forms/Create';
import DepartmentMembersForm from 'components/Department/Forms/AddMembers';

const AddMenu = React.createClass({

    _handleFormDone() {
        ModalActions.close();
    },

    _handleMenuSelect(payload) {
        const { id, onSelect } = this.props;
        let isSubmitted = false;

        switch (payload.value) {
            case 'user':
                Metrika.send('Структура', 'Меню добавления', 'Добавить сотрудника');
                UserModalActions.create({ department: id });
                break;

            case 'invite':
                Metrika.send('Структура', 'Меню добавления', 'Пригласить сотрудников');
                InviteModalActions.create({ department: id });
                break;

            case 'department':
            case 'subdepartment':

                Metrika.send(
                    'Структура', 'Меню добавления',
                    payload.value === 'subdepartment' ? 'Создать подотдел' : 'Создать отдел'
                );
                timer.start('create_department');

                ModalActions.open({
                    title: payload.value === 'subdepartment' ?
                        i18n('department.new_subdepartment') :
                        i18n('department.new_department'),
                    component: DepartmentForm,
                    props: {
                        parentId: id,
                        onSubmit: () => {
                            isSubmitted = true;
                            this._handleFormDone();
                        },
                        onCancel: () => {
                            timer.clear('create_department');
                            this._handleFormDone();
                        },
                    },
                    onClose: () => {
                        if (!isSubmitted) {
                            timer.clear('create_department');
                        }
                    },
                });
                break;

            case 'existing_user':
                Metrika.send('Структура', 'Меню добавления', 'Переместить сотрудника');

                ModalActions.open({
                    component: DepartmentMembersForm,
                    props: {
                        title: i18n('department.add_user', {
                            department: DepartmentStore.getName(id),
                        }),
                        id,
                        type: 'users',
                        multiple: false,
                        onSubmit: this._handleFormDone,
                        onCancel: this._handleFormDone,
                    },
                });
                break;

            case 'existing_departments':
                Metrika.send('Структура', 'Меню добавления', 'Переместить отделы');

                ModalActions.open({
                    component: DepartmentMembersForm,
                    props: {
                        title: i18n('department.add_departments', {
                            department: DepartmentStore.getName(id),
                        }),
                        id,
                        type: 'departments',
                        onSubmit: this._handleFormDone,
                        onCancel: this._handleFormDone,
                    },
                });
                break;

            /* no default */
        }

        if (onSelect) {
            onSelect(payload);
        }
    },

    render() {
        const department = DepartmentStore.get(this.props.id);

        if (!department) {
            return null;
        }

        const option = {
            addUser: {
                text: i18n('department.add_menu.user'),
                val: 'user',
                disabled: !PermissionStore.contains(Permissions.ADD_USERS, department),
            },
            inviteUsers: {
                text: i18n('department.add_menu.invite_user'),
                val: 'invite',
            },
            addDepartment: {
                text: i18n('department.add_menu.department'),
                val: 'department',
                disabled: !PermissionStore.contains(Permissions.ADD_DEPARTMENTS, department),
            },
            addSubdepartment: {
                text: i18n('department.add_menu.subdepartment'),
                val: 'subdepartment',
                disabled: !PermissionStore.contains(Permissions.ADD_DEPARTMENTS, department),
            },
            addExistingUser: {
                text: i18n('department.add_menu.existing_user'),
                val: 'existing_user',
                disabled: !PermissionStore.contains(Permissions.ADD_USERS, department) &&
                    !PermissionStore.contains(Permissions.INVITE_USERS, department),
            },
            addExistingDepartments: {
                text: i18n('department.add_menu.existing_departments'),
                val: 'existing_departments',
                disabled: !PermissionStore.contains(Permissions.ADD_DEPARTMENTS, department),
            },
        };

        const optionList = [{
            group: [
                option.addUser,
                PermissionStore.contains(Permissions.INVITE_USERS, department) && option.inviteUsers,
                department.isRoot() ? option.addDepartment : option.addSubdepartment,
            ].filter(Boolean),
        }, {
            group: [
                option.addExistingUser,
                option.addExistingDepartments,
            ],
        }];

        return <Menu options={optionList} onSelect={this._handleMenuSelect} />;
    },
});

AddMenu.propTypes = {
    id: PropTypes.string,
    onSelect: PropTypes.func,
};

export default AddMenu;
