import _ from 'lodash';
import PropTypes from 'prop-types';
import React from 'react';
import PureRenderMixin from 'react-addons-pure-render-mixin';
import StoreMixin from 'lib/StoreMixin';
import Url from 'lib/Url';
import { i18n } from 'lib/i18n';

import DepartmentStore from 'stores/Departments';
import PermissionStore from 'stores/Permissions';
import AuthStore from 'stores/Auth';

import Link from 'ui/Link';
import Unit from 'ui/Unit';

import Hoverable from 'ui/Hoverable';
import HoverCard from 'components/User/HoverCard';
import Edit from './Controls/Edit';

const Header = React.createClass({

    mixins: [StoreMixin, PureRenderMixin],

    getStoreState() {
        return {
            department: DepartmentStore.get(this.props.id),
        };
    },

    componentDidMount() {
        this.subscribe([
            DepartmentStore,
        ]);
    },

    render() {
        const { id } = this.props;
        const { department } = this.state;
        let component = {};

        if (!department) {
            return null;
        }

        const organization = AuthStore.getOrganization();
        const description = department.getDescription();

        if (description) {
            component.description = (
                <div className="department-description section-header__description">
                    {description}
                </div>
            );
        }

        let head = department.getHead();

        if (!head && department.isRoot()) {
            head = organization.getHead();
        }

        if (head) {
            component.headHoverCard = <HoverCard user={head} />;
            component.head = (
                <div className="department-head">
                    <Hoverable hoverContent={component.headHoverCard}>
                        <Link to={head.getUrl()}>
                            <Unit
                                avatar={head.getAvatar()}
                                title={head.getAlphabeticalName()}
                                description={i18n('department.head')}
                                type={head.getType()}
                                role={head.getRole()}
                                tooltip={false}
                            />
                        </Link>
                    </Hoverable>
                </div>
            );
        }

        const email = department.getEmail();

        if (email) {
            component.contacts = (
                <div className="department-contacts list">
                    <div className="list-item">
                        <div className="list-item__key">
                            {i18n('department.email')}:
                        </div>
                        <div className="list-item__value" title={email}>
                            <Link to={Url.getEmail(email)} target="_blank">{email}</Link>
                        </div>
                    </div>
                </div>
            );
        }

        const isRoot = department.isRoot();

        if (isRoot) {
            component = _.pick(component, 'contacts');
        }

        if (component.description || component.contacts || component.head) {
            component.info = (
                <div className="department-info section-header__info">
                    {component.description}
                    {component.contacts}
                    {component.head}
                </div>
            );
        }

        const title = isRoot ? organization.getName() : department.getName();

        if (PermissionStore.allowsDepartmentEditing(department) && (!isRoot || AuthStore.hasOwnedDomains())) {
            component.editControl = (
                <Edit
                    id={id}
                    title={i18n('department.action_description.edit', {
                        department_name: department.getName(),
                    })}
                />
            );
        } else if (!component.info && !title) {
            // незаполненная нередактируемая карточка
            // отображается без шапки
            return null;
        }

        component.title = (
            <h2 className="department-header__title section-header__title" title={title}>
                {title}
            </h2>
        );

        return (
            <div className="department-header section-header">
                {component.editControl}
                {component.title}
                {component.info}
            </div>
        );
    },

});

Header.propTypes = {
    id: PropTypes.string,
};

export default Header;
