import React from 'react';
import classSet from 'classnames';
import { Spin } from 'lego-on-react';
import StoreMixin from 'lib/StoreMixin';
import DnsStore from 'stores/Dns';
import { i18n } from 'lib/i18n';
import Metrika from 'lib/metrika';
import { getContent, getPriority } from 'lib/dns';

import ModalActions from 'actions/Modal';
import DnsActions from 'actions/Dns';
import submit from 'services/submit';

import EditForm from './Forms/Edit';

export default React.createClass({

    mixins: [StoreMixin],

    getStoreState() {
        const records = DnsStore.getList()
            .filter(record => record.type !== 'SOA');

        return {
            records,
        };
    },

    componentDidMount() {
        this.subscribe(DnsStore);
    },

    _renderItem(record, index) {
        const fieldClass = classSet({
            'dns-list__field': true,
            'dns-list__field_even': index % 2 === 1,
        });

        const controls = {};

        if (this.props.editable) {
            controls.edit = (
                <div
                    className="dns-list__edit"
                    onClick={this._handleEdit}
                    data-id={record.id}
                />
            );
            controls.remove = (
                <div
                    className="dns-list__delete"
                    onClick={this._handleRemove}
                    data-id={record.id}
                />
            );
        }

        return (
            <div className={fieldClass} key={record.id}>
                <div className="dns-list__host">
                    {record.name}
                </div>
                <div className="dns-list__type">
                    {record.type}
                </div>
                <div className="dns-list__value">
                    {getContent(record)}
                </div>
                <div className="dns-list__priority">
                    {getPriority(record)}
                </div>
                {controls.edit}
                {controls.remove}
            </div>
        );
    },

    _handleEdit(event) {
        const { domain } = this.props;
        const { id } = event.target.dataset;

        Metrika.send('DNS', 'Cписок', 'Клик в редактировать');

        ModalActions.open({
            component: EditForm,
            props: {
                domain: domain && domain.getName(),
                recordId: id,
                recordType: DnsStore.getRecordType(id),
                onSubmit: this._handleFormDone,
                onCancel: this._handleFormDone,
            },
        });
    },

    _handleFormDone() {
        ModalActions.close();
    },

    _handleRemove(event) {
        const { domain } = this.props;
        const { id } = event.target.dataset;

        Metrika.send('DNS', 'Cписок', 'Клик в удалить');

        ModalActions.confirm({
            title: i18n('dns_record.message.confirm_removal'),
            confirmButtonText: i18n('common.action.remove'),
            onConfirm() {
                submit(DnsActions.removeRecord(domain && domain.getName(), id), {
                    success: i18n('dns_record.status.removed'),
                    failure: i18n('dns_record.status.failed_to_remove'),
                });
            },
        });
    },

    render() {
        const { records } = this.state;

        if (this.props.busy) {
            return (
                <div className="dns-list__status">
                    {'\u00a0'}
                    <Spin
                        progress
                        size="m"
                        position="center"
                    />
                </div>
            );
        }

        if (!records.length) {
            return (
                <div className="dns-list__status">
                    {i18n('dns_settings.no_records')}
                </div>
            );
        }

        return (
            <div className="dns-list">
                <div className="dns-list__header">
                    <div className="dns-list__host">
                        {i18n('dns_record.host')}
                    </div>
                    <div className="dns-list__type">
                        {i18n('dns_record.type')}
                    </div>
                    <div className="dns-list__value">
                        {i18n('dns_record.content')}
                    </div>
                    <div className="dns-list__priority">
                        {i18n('dns_record.priority')}
                    </div>
                </div>
                {records.map(this._renderItem)}
            </div>
        );
    },

});
