import _ from 'lodash';
import { i18n } from 'lib/i18n';
import template from 'lib/template';

const CNAME_CONTENT_MAP = {
    ru: 0, com: 1, 'com.tr': 2,
};

const htmlFileTemplate = [
    '<html>',
    '    <head>',
    '        <meta http-equiv="Content-Type" content="text/html; charset=UTF-8">',
    '    </head>',
    '    <body>Verification: ${code}</body>',
    '</html>',
].join('\n');

const confirmationOptions = {
    pdd: {
        file: item => [
            {
                key: 'domain.confirmation_methods.file.description',
                params: {
                    file_name: item.filename,
                    content: item.content,
                },
            },
        ],
        cname: item => [
            {
                key: 'domain.confirmation_methods.cname.description',
                params: {
                    subdomain: item.subdomain,
                    value: item.contents[CNAME_CONTENT_MAP[item.tld] || 0],
                },
            },
        ],
        delegate: item => [
            {
                key: 'domain.confirmation_methods.delegate.description.a',
            },
            {
                key: 'domain.confirmation_methods.delegate.description.b',
                params: {
                    server: item.servers[1],
                },
            },
            {
                key: 'domain.confirmation_methods.delegate.description.c',
                params: {
                    help_url: item.help,
                },
            },
        ],
        whois: item => [
            {
                key: 'domain.confirmation_methods.whois.description.a',
                params: {
                    email: item.strings[0],
                    domain: item.domain,
                },
            },
            {
                key: 'domain.confirmation_methods.whois.description.b',
                params: {
                    email: formatValue(item.strings[1]),
                },
            },
        ],
    },
    webmaster: {
        dns: item => [
            {
                key: 'domain.confirmation_methods.webmaster.dns.b',
            },
            {
                type: 'code',
                key: 'domain.confirmation_methods.webmaster.dns.a',
                template: 'yandex-verification: ${code}',
                params: {
                    domain: item.domain,
                    code: item.code,
                },
            },
            {
                key: 'domain.confirmation_methods.webmaster.dns.b2',
            },
            {
                key: 'domain.confirmation_methods.webmaster.dns.c',
                params: {
                    button_text: item.buttonText,
                },
            },
        ],
        html_file: item => [
            {
                type: 'code',
                key: 'domain.confirmation_methods.webmaster.html_file.a',
                template: htmlFileTemplate,
                params: {
                    code: item.code,
                },
            },
            {
                key: 'domain.confirmation_methods.webmaster.html_file.b',
                params: {
                    domain: item.domain,
                    code: item.code,
                },
                withoutFormatting: true,
            },
            {
                key: 'domain.confirmation_methods.webmaster.html_file.c',
                params: {
                    button_text: item.buttonText,
                },
            },
        ],
        meta_tag: item => [
            {
                type: 'code',
                key: 'domain.confirmation_methods.webmaster.meta_tag.a',
                template: '<meta name="yandex-verification" content="${code}" />',
                params: {
                    code: item.code,
                },
            },
            {
                key: 'domain.confirmation_methods.webmaster.meta_tag.b',
            },
            {
                key: 'domain.confirmation_methods.webmaster.meta_tag.c',
                params: {
                    button_text: item.buttonText,
                },
            },
        ],
        whois: item => [
            {
                key: 'domain.confirmation_methods.webmaster.whois.0',
            },
            {
                key: 'domain.confirmation_methods.webmaster.whois.a',
                params: {
                    passport_url: item.confirmEmailUrl || '',
                },
            },
            {
                key: 'domain.confirmation_methods.webmaster.whois.b',
                params: {
                    button_text: item.buttonText,
                },
            },
        ],
    },
};

// вставка потенциальных переносов перед небуквенными символами
// в длинных технических значениях, таких как домены, названия файлов и пр.
function formatValue(s) {
    return s ? `<span class="value">${s.replace(/([@\.\+\-])/g, '<wbr>$1')}</span>` : '';
}

export default function(confirmationData) {
    const { method } = confirmationData;
    const [origin, methodName] = method.split('.');
    const getOptionInfo = _.get(confirmationOptions, [origin, methodName]);

    if (!getOptionInfo) {
        return [];
    }

    return getOptionInfo(confirmationData)
        .map(contentItem => {
            if (contentItem.type === 'code') {
                const code = template.build(contentItem.template, contentItem.params);
                const title = contentItem.key ? `${i18n(contentItem.key, contentItem.params)}<br/>` : '';

                return `${title}<pre class="code">${_.escape(code)}</pre>`;
            }

            const allParamsUndefined = !_.isEmpty(contentItem.params) &&
                !_.some(contentItem.params, value => value !== undefined);

            if (allParamsUndefined) {
                return null;
            }

            if (!contentItem.withoutFormatting) {
                Object.keys(contentItem.params || {}).forEach(key => {
                    if (key.indexOf('_url') === -1) {
                        contentItem.params[key] = formatValue(contentItem.params[key]);
                    }
                });
            }

            return i18n(contentItem.key, contentItem.params);
        })
        .filter(Boolean);
}
