import React from 'react';
import PureRenderMixin from 'react-addons-pure-render-mixin';
import classSet from 'classnames';
import { i18n } from 'lib/i18n';
import timer from 'lib/timer';
import { RouteTiming } from 'lib/rum';
import getCurrentOs from 'lib/getCurrentOs';
import ConfigStore from 'stores/Config';

import Header from 'components/Header';
import Link from 'ui/Link';

import './index.css';
import './items.css';
import './platforms.css';
import template from 'client/lib/template';

const DOWNLOAD_ITEMS = [{
    id: 'yamb',
    metrika: 'Ямб',
    urlParams: {
        platform: getCurrentOs(),
    },
}, {
    id: 'mail',
    metrika: 'Почта',
}, {
    id: 'disk',
    metrika: 'Диск',
}, {
    id: 'tracker',
    metrika: 'Трекер',
}, {
    id: 'browser',
    metrika: 'Браузер',
}];

const PLATFORM_ORDER = ['ios', 'android', 'windows', 'pc'];

const Downloads = React.createClass({

    mixins: [PureRenderMixin],

    componentDidMount() {
        RouteTiming.end();

        document.body.className = 'loaded connect-app';
        document.body.dataset.section = 'connect-downloads';

        timer.stop('first_run');
    },

    _renderItemApps(item) {
        const apps = ConfigStore.getRelatedApps(item.id);

        if (!apps) {
            return null;
        }

        const list = PLATFORM_ORDER.map(platform => {
            let appUrl = apps[platform];

            if (!appUrl) {
                return null;
            }

            if (item.urlParams) {
                appUrl = template.build(appUrl, item.urlParams);
            }

            const className = classSet({
                'app-download': true,
                [`platform__${platform}`]: true,
            });

            const metrika = ['Приложения', item.metrika, platform];

            if (platform === 'pc') {
                metrika.push(getCurrentOs());
            }

            return (
                <Link
                    className={className}
                    to={appUrl}
                    target="_self"
                    metrika={metrika}
                    key={platform}
                />
            );
        }).filter(Boolean);

        if (!list.length) {
            return null;
        }

        return <div className="app-download__list">{list}</div>;
    },

    _renderItem(item) {
        const url = item.url || ConfigStore.getRelatedServiceUrl(item.id);

        const title = item.title ||
            i18n(`downloads.${item.id}.title`) ||
            i18n(`related_services.${item.id}`);

        const description = (i18n(`downloads.${item.id}.description`) || '')
            .replace(/\n/g, '<br>');

        const className = classSet({
            'download-item': true,
            [`download-item__${item.id}`]: true,
        });

        return (
            <div className={className} key={item.id}>
                <div className="download-item__frame">
                    <div className="download-item__icon" />
                    <div className="download-item__title">
                        <Link
                            className="download-item__link"
                            to={url}
                            target="_self"
                            metrika={['Приложения', item.metrika, 'web']}
                        >
                            {title}
                        </Link>
                    </div>
                    <div
                        className="download-item__description"
                        dangerouslySetInnerHTML={{ __html: description }}
                    />
                    <div className="download-item__apps">
                        {this._renderItemApps(item)}
                    </div>
                </div>
            </div>
        );
    },

    render() {
        return (
            <div className="downloads layout-container">
                <Header />
                <div className="downloads-header">
                    <div className="downloads-header__title">
                        {i18n('downloads.title')}
                    </div>
                </div>
                <div className="downloads-body" data-grid="4">
                    {DOWNLOAD_ITEMS.map(this._renderItem)}
                </div>
            </div>
        );
    },

});

export default Downloads;
