import PropTypes from 'prop-types';
import React from 'react';
import PureRenderMixin from 'react-addons-pure-render-mixin';
import Avatar from 'ui-components/lib/Avatar';
import Url from 'lib/Url';
import { i18n } from 'lib/i18n';

import GroupStore from 'stores/Groups';
import PermissionStore from 'stores/Permissions';

import ListItem from 'ui/ListItem';
import Link from 'ui/Link';

import Hoverable from 'ui/Hoverable';
import HoverCard from 'components/User/HoverCard';
import Edit from './Controls/Edit';

const Header = React.createClass({

    mixins: [PureRenderMixin],

    render() {
        const { id } = this.props;
        const group = GroupStore.get(id);
        const component = {};

        if (!group) {
            return null;
        }

        const description = group.getDescription();

        if (description) {
            component.description = (
                <div className="section-header__description">
                    {description}
                </div>
            );
        }

        const email = group.getEmail();

        if (email) {
            component.email = (
                <ListItem key="group-email" label={`${i18n('group.group_email')}:`}>
                    <Link to={Url.getEmail(email)} title={email} target="_blank">{email}</Link>
                </ListItem>
            );
        }

        const author = group.getAuthor();
        const authorName = author ? author.getName() : null;

        if (authorName) {
            component.authorHoverCard = <HoverCard user={author} />;
            component.author = (
                <ListItem key="group-author" label={`${i18n('group.author')}:`}>
                    <Hoverable hoverContent={component.authorHoverCard}>
                        <Link to={author.getUrl()} title={authorName}>{authorName}</Link>
                    </Hoverable>
                </ListItem>
            );
        }

        const creationDate = group.getReadableCreationDate();

        if (creationDate) {
            component.creationDate = (
                <ListItem
                    key="group-creation-date"
                    value={creationDate}
                    label={`${i18n('group.creation_date')}:`}
                />
            );
        }

        if (component.email || component.author || component.creationDate) {
            component.info = (
                <div className="section-header__info list">
                    {component.email}
                    {component.author}
                    {component.creationDate}
                </div>
            );
        }

        if (PermissionStore.allowsGroupEditing(group)) {
            component.editControl = (
                <Edit
                    id={id}
                    groupName={i18n('group.action_description.edit', {
                        group_name: group.getName(),
                    })}
                />
            );
        }

        return (
            <div className="group-header section-header">
                {component.editControl}
                <div className="group-header__avatar">
                    <Avatar url={group.getAvatar()} size="xl" />
                </div>
                <div className="group-header__content">
                    <h2 className="section-header__title" title={group.getName()}>
                        {group.getName()}
                    </h2>
                    {component.description}
                    {component.info}
                </div>
            </div>
        );
    },
});

Header.propTypes = {
    id: PropTypes.string,
};

export default Header;
