import ApplicationStore from 'stores/Application';
import AuthStore from 'stores/Auth';
import PermissionStore from 'stores/Permissions';
import Url from 'lib/Url';
import { i18n } from 'lib/i18n';

import GroupModalActions from 'actions/Modal.Group';

function matchesStateTypes(types) {
    return types.indexOf(ApplicationStore.getStateType()) !== -1;
}

const MENU_SECTIONS = [
    {
        index: {
            text: i18n('common.groups'),
            metrika: 'Команды',
            icon: 'teams',
            url: () => Url.getGroupListPath('all'),
            matches: () => matchesStateTypes([
                'groups',
                'groups/admins/users',
                'groups/admins/departments',
                'groups/admins/groups',
                'groups/members/users',
                'groups/members/departments',
                'groups/members/groups',
            ]),
            actions: {
                add: {
                    callback: () => GroupModalActions.create(),
                    description: i18n('group.action_description.add'),
                },
            },
            disabled: () => !PermissionStore.allowsGroupsAdd(),
            available: () =>
                AuthStore.isIncompleteOrganization() ||
                    PermissionStore.allowsGroupsAdd() ||
                    PermissionStore.allowsOrganizationStructureEditing(),

        },
        subsections: [
            {
                text: i18n('sections.groups.admin'),
                metrika: 'Я администратор',
                url: () => Url.getGroupListPath('admin'),
                matches: () => matchesStateTypes([
                    'groups/admin',
                    'groups/admin/admins/users',
                    'groups/admin/admins/departments',
                    'groups/admin/admins/groups',
                    'groups/admin/members/users',
                    'groups/admin/members/departments',
                    'groups/admin/members/groups',
                ]),
                available: () => AuthStore.isInternal(),
                disabled: () => AuthStore.isIncompleteOrganization(),
            },
            {
                text: i18n('sections.groups.member'),
                metrika: 'Я участник',
                url: () => Url.getGroupListPath('member'),
                matches: () => matchesStateTypes([
                    'groups/member',
                    'groups/member/admins/users',
                    'groups/member/admins/departments',
                    'groups/member/admins/groups',
                    'groups/member/members/users',
                    'groups/member/members/departments',
                    'groups/member/members/groups',
                ]),
                available: () => AuthStore.isInternal(),
                disabled: () => AuthStore.isIncompleteOrganization(),
            },
        ],
    },
    {
        index: {
            text: i18n('services.plans_and_services'),
            metrika: 'Тарифы и сервисы',
            icon: 'services',
            url: () => Url.getDashboardUrl(),
            disabled: () => !PermissionStore.allowsManageServices(),
            available: () => PermissionStore.allowsManageServices() || AuthStore.isIncompleteOrganization(),
        },
        subsections: [
            {
                text: i18n('services.name.tracker'),
                metrika: 'Трекер',
                url: () => Url.getLocation(Url.getServiceSettingsUrl('tracker')),
                available: () => PermissionStore.allowsTrackerManagement(),
                disabled: false,
            },
            {
                text: i18n('sections.services.mail'),
                metrika: 'Почта',
                url: () => Url.getLocation(Url.getServiceSettingsUrl('mail')),
                available: () => PermissionStore.allowsMailManagement(),
                disabled: false,
            },
        ],
    },
    {
        index: {
            text: i18n('sections.balance'),
            metrika: 'Баланс',
            icon: 'billing',
            url: () => Url.getLocation(Url.getPath('balance')),
            disabled: () => !PermissionStore.allowsPay(),
            available: () => PermissionStore.allowsPay() || AuthStore.isIncompleteOrganization(),
        },
    },
    {
        index: {
            text: i18n('sections.organization_profile'),
            metrika: 'Профиль организации',
            icon: 'organization',
            url: () => Url.getLocation(Url.getPath('profile')),
            disabled: () => !PermissionStore.allowsOrganizationProfileEditing(),
            available: () => AuthStore.isIncompleteOrganization() || PermissionStore.allowsOrganizationProfileEditing(),
        },
        subsections: [
            {
                text: i18n('sections.customization.admins'),
                metrika: 'Администраторы',
                url: () => Url.getContextPath('customization', 'admins'),
                matches: () => matchesStateTypes([
                    'customization/admins',
                ]),
                available: () => PermissionStore.allowsChangeRole() || AuthStore.isIncompleteOrganization(),
                disabled: () => !PermissionStore.allowsChangeRole(),
            },
            {
                text: i18n('sections.customization.dns'),
                metrika: 'Управление DNS',
                url: () => Url.getContextPath('customization', 'dns'),
                available: () => AuthStore.isIncompleteOrganization() || PermissionStore.allowsDNSEditing(),
                disabled: () => !PermissionStore.allowsDNSEditing(),
            },
            {
                text: i18n('sections.customization.domains'),
                metrika: 'Домены',
                url: () => Url.getContextPath('domains'),
                matches: () => matchesStateTypes([
                    'domains',
                ]),
                available: () => PermissionStore.allowsDomainsEditing(),
                actions: {
                    add: {
                        callback: () => Url.open(Url.getLocation(Url.getPath('services/webmaster#add-domain'))),
                        description: i18n('domain.action_description.add'),
                    },
                },
            },
            {
                text: i18n('sections.customization.general'),
                metrika: 'Дополнительные настройки',
                url: () => Url.getContextPath('customization', 'general'),
                available: () => AuthStore.hasOwnedDomains() || AuthStore.isIncompleteOrganization(),
                disabled: () => !AuthStore.hasOwnedDomains(),
            },
        ],
    },
];

export default MENU_SECTIONS;
