import PropTypes from 'prop-types';
import React from 'react';
import Menu from 'react-bem-components/lib/Menu';
import { Permissions } from 'constants/Common';
import Url from 'lib/Url';
import { i18n } from 'lib/i18n';
import StoreMixin from 'lib/StoreMixin';

import ConfigStore from 'stores/Config';
import PermissionStore from 'stores/Permissions';
import OrganizationStore from 'stores/Organizations';

import UIActions from 'actions/UI';
import ModalActions from 'actions/Modal';

import UpdateSection from 'components/OrgProfile/UpdateSection';
import RemoveOrg from '../Components/Remove';

const MENU_OPTIONS = [{
    text: i18n('organization_profile.edit_menu.edit'),
    val: 'update',
    available: () => PermissionStore.contains(Permissions.EDIT_ORGANIZATION),
}, {
    text: i18n('organization_profile.edit_menu.remove'),
    val: 'remove',
}];

const deleteStatuses = {
    ALLOWED: 'allowed',
    FORBIDDEN: 'forbidden',
};

const EditMenu = React.createClass({

    mixins: [StoreMixin],

    getStoreState() {
        let deleteStatus = deleteStatuses.ALLOWED;

        if (!PermissionStore.contains(Permissions.DELETE_ORGANIZATION)) {
            deleteStatus = deleteStatuses.FORBIDDEN;
        }

        return { deleteStatus };
    },

    componentDidMount() {
        this.subscribe([PermissionStore, OrganizationStore]);
    },

    _handleMenuSelect(payload) {
        const { onSelect } = this.props;

        switch (payload.value) {
            case 'update':
                UIActions.editOrgProfile({
                    component: UpdateSection,
                });
                break;

            case 'remove':
                this._onDeleteClick();
                break;

        /* no default */
        }

        if (onSelect) {
            onSelect(payload);
        }
    },

    _handleFormDone() {
        ModalActions.close();
    },

    _onDeleteClick() {
        const { organization } = this.props;
        const { deleteStatus } = this.state;

        if (deleteStatus === deleteStatuses.FORBIDDEN) {
            ModalActions.confirm({
                title: i18n('organization_profile.delete_org_popup.title.forbidden', {
                    org_name: organization.getName(),
                }),
                message: i18n('organization_profile.delete_org_popup.message.forbidden'),
                confirmButtonText: i18n('organization_profile.delete_org_popup.button.send_request'),
                onConfirm: () => this._redirectToSupport(),
            });

            return;
        }

        ModalActions.open({
            component: RemoveOrg,
            props: {
                title: i18n('organization_profile.delete_org_popup.title.allowed', {
                    org_name: organization.getName(),
                }),
                onCancel: this._handleFormDone,
                orgId: organization.getId(),
            },
        });
    },

    _redirectToSupport() {
        Url.open(ConfigStore.getHelpUrl('ui.help.feedback'));
    },

    render() {
        const menuOptions = MENU_OPTIONS
            .filter(item => item.available === undefined || item.available());

        return <Menu options={menuOptions} onSelect={this._handleMenuSelect} />;
    },

});

EditMenu.propTypes = {
    onSelect: PropTypes.func,
};

export default EditMenu;
