import React from 'react';
import PureRenderMixin from 'react-addons-pure-render-mixin';
import Spin from 'react-bem-components/lib/Spin';
import { i18n, toHtml } from 'lib/i18n';
import Url from 'lib/Url';
import timer from 'lib/timer';
import Metrika from 'lib/metrika';
import { RouteTiming } from 'lib/rum';

import Icon from 'ui/Icon';

import ConfigStore from 'stores/Config';
import OrgSetupActions from 'actions/OrgSetup';
import ErrorActions from 'actions/App/Error';
import Promo from './Promo';
import DetailsForm from './Forms/Details';

import './index.css';

const TRACKING_FIELD_MAP = {
    name: 'Название организации',
    country: 'Страна',
    language: 'Язык',
    domain: 'Домен',
    email: 'Личная почта',
};

function track() {
    const args = Array.prototype.slice.call(arguments);

    args.unshift('Дорегистрация', ConfigStore.get('app.tld'));

    Metrika.send.apply(Metrika, args);
}

// DIR-2533
// проставляем куку, чтобы первое время читать данные пользователя из мастер-базы
function sync() {
    document.cookie = `connect_first_load=yes; path=/; expires=${
        (new Date(Date.now() + 5 * 60 * 1000)).toUTCString()}`;
}

const SOURCE_PRESET_MAP = {
    tracker_promo: 'tracker',
};

function getPreset() {
    return Url.getQueryParam('preset') || SOURCE_PRESET_MAP[Url.getQueryParam('source')];
}

const OrgSetup = React.createClass({

    mixins: [PureRenderMixin],

    getInitialState() {
        return {
            preset: getPreset(),
            data: 'processing',
            form: 'blank',
        };
    },

    componentDidMount() {
        const { preset } = this.state;

        document.body.className = 'loaded connect-app';
        document.body.dataset.section = 'connect-setup';

        track('Запущена');

        timer.stop('first_run');

        RouteTiming.end();

        this._initialAction = OrgSetupActions.submitInitialData({ preset })
            .then(() => {
                sync();
                track('Данные первого шага из Паспорта отправлены');

                if (preset === 'tracker') {
                    Metrika.reachGoal('AutoActivatedTrackerFromRegistration');
                }

                this.setState({ data: 'pushed' });
                this._tryNext();
            })
            .catch(() => {
                this.setState({ data: 'failed' });
                this._tryNext();
            });
    },

    _submitForm() {
        track('Форма заполнения информации об организации', 'Отправить');

        return this._initialAction;
    },

    _setFormReady() {
        track('Форма заполнения информации об организации', 'Отправлена');
        this.setState({ form: 'ready' });
        this._tryNext();
    },

    _skipForm() {
        track('Форма заполнения информации об организации', 'Не требует заполнения');
        this.setState({ form: 'skipped' });
        this._tryNext();
    },

    _handleChange(field) {
        track(
            'Форма заполнения информации об организации',
            'Пользователь меняет значение',
            TRACKING_FIELD_MAP[field] || field
        );
    },

    _tryNext() {
        if (this.state.form === 'ready' || this.state.form === 'skipped') {
            switch (this.state.data) {
                case 'pushed':
                    track('Завершена');
                    location.href =
                        Url.getSafeRedirectUrl(Url.getQueryParam('retpath')) ||
                    (this.state.preset ? Url.getPath('admin', 'subscription', 'services') : '/');
                    break;
                case 'failed':
                    track('Ошибка при создании организации');
                    ErrorActions.showError();
                    break;
            }
        }
    },

    render() {
        return (
            <div className="app">
                <div className={`app__body data_${this.state.data} form_${this.state.form}`}>
                    <Promo serviceId={this.state.preset} />
                    <DetailsForm
                        onSubmit={this._submitForm}
                        onReady={this._setFormReady}
                        onSkip={this._skipForm}
                        onChange={this._handleChange}
                    />
                    <div className="status-item status-item_processing">
                        <Spin visible size="xl" />
                        <div
                            className="status-item__caption"
                            dangerouslySetInnerHTML={{
                                __html: toHtml(i18n('organization_setup.status.in_progress')),
                            }}
                        />
                    </div>
                    <div className="status-item status-item_failed">
                        <Icon type="status-cross" size="xl" />
                        <div
                            className="status-item__caption"
                            dangerouslySetInnerHTML={{
                                __html: toHtml(i18n('common.status.failed')),
                            }}
                        />
                    </div>
                </div>
            </div>
        );
    },

});

export default OrgSetup;
