import React from 'react';
import classSet from 'classnames';
import { Button } from 'lego-on-react';
import { Permissions } from 'constants/Common';
import StoreMixin from 'lib/StoreMixin';
import { i18n, formatCurrency, formatDate } from 'lib/i18n';

import PermissionStore from 'stores/Permissions';
import SubscriptionStore from 'stores/Subscription';
import SubscriptionActions from 'actions/Subscription';

import './index.css';

const PAYMENT_EXPECTATION_TIME = 0; // days

function toDate(date) {
    return date instanceof Date ? date : new Date(date);
}

const PaymentNotification = React.createClass({

    mixins: [StoreMixin],

    getStoreState() {
        // return {
        //     plan: 'free',
        //     balance: -10000,
        //     paymentDueDate: '2018-04-02',
        //     debtStartDate: '2018-04-01',
        //     daysToBlocked: 0
        // };

        return {
            plan: SubscriptionStore.getPlan(),
            balance: SubscriptionStore.getBalance(),
            paymentDueDate: SubscriptionStore.get('current.payment_due_date'),
            debtStartDate: SubscriptionStore.get('current.debt_start_date'),
            daysToBlocked: SubscriptionStore.get('current.days_until_disable_by_debt'),
        };
    },

    componentDidMount() {
        this.subscribe([SubscriptionStore, PermissionStore]);
    },

    _shouldRender() {
        if (!PermissionStore.contains(Permissions.PAY)) {
            return false;
        }

        const { balance, paymentDueDate, debtStartDate } = this.state;

        if (balance === null || balance === undefined || balance >= 0 || !paymentDueDate) {
            return false;
        }

        const referenceDate = new Date();

        referenceDate.setDate(referenceDate.getDate() - PAYMENT_EXPECTATION_TIME);

        if (!debtStartDate || referenceDate.getTime() < toDate(debtStartDate).getTime()) {
            return false;
        }

        return true;
    },

    _isBlocked() {
        return this.state.daysToBlocked === 0;
    },

    _hasWarning() {
        const { daysToBlocked } = this.state;

        return typeof daysToBlocked === 'number' && daysToBlocked <= 14;
    },

    render() {
        const { balance, paymentDueDate } = this.state;

        if (!this._shouldRender()) {
            return null;
        }

        const type = this._isBlocked() ? 'blocked' : 'will_block';

        const content = i18n(`balance.notification.${type}.description`, {
            amount: formatCurrency(balance, 'RUB'),
            date: formatDate(paymentDueDate, 'day-and-month'),
        });

        const classes = classSet({
            'payment-notification': true,
            'payment-notification_warning': this._hasWarning(),
        });

        return (
            <div className={classes}>
                <div className="payment-notification__content">
                    <h2 className="payment-notification__title">
                        {i18n(`balance.notification.${type}.title`, {
                            date: formatDate(paymentDueDate, 'day-and-month'),
                        })}
                    </h2>
                    <div
                        className="payment-notification__description"
                        dangerouslySetInnerHTML={{ __html: content }}
                    />
                </div>
                <div className="payment-notification__options">
                    <Button
                        text={i18n('balance.notification.pay_button')}
                        onClick={() => SubscriptionActions.toPayment()}
                        size="m"
                        theme="normal"
                    />
                </div>
            </div>
        );
    },

});

export default PaymentNotification;
