import _ from 'lodash';
import React from 'react';

import PureRenderMixin from 'react-addons-pure-render-mixin';
import { Button } from 'lego-on-react';
import StoreMixin from 'lib/StoreMixin';
import UIStore from 'stores/UI';

import StatusBarActions from 'actions/StatusBar';

import Icon from 'ui/Icon';

import './index.css';

const BASE_STATE = {
    message: undefined,
    confirmText: undefined,
    rejectText: undefined,
    onConfirm: undefined,
    onReject: undefined,
    isEnabled: false,
};

const StatusBar = React.createClass({
    mixins: [StoreMixin, PureRenderMixin],

    getStoreState() {
        const statusBarData = UIStore.getStatusBarData('statusBar');

        return _.extend({}, BASE_STATE, statusBarData, { isEnabled: Boolean(statusBarData) });
    },

    componentDidMount() {
        this.subscribe(UIStore);
    },

    componentWillUpdate(nextProps, nextState) {
        if (this.state.isEnabled && !nextState.isEnabled) {
            this._removeListeners();
        }
    },

    componentDidUpdate(prevProps, prevState) {
        if (!prevState.isEnabled && this.state.isEnabled) {
            this._addListeners();
        }

        if (_.isEqual(this.state, prevState)) {
            this.setState(this.getStoreState());
        }
    },

    _addListeners() {
        if (this.refs.statusBar) {
            this.refs.statusBar.addEventListener('click', this._onStatusBarClick);
        }
        document.body.addEventListener('click', this._off);
    },

    _removeListeners() {
        document.body.removeEventListener('click', this._off);

        if (this.refs.statusBar) {
            this.refs.statusBar.removeEventListener('click', this._onStatusBarClick);
        }
    },

    _onStatusBarClick(event) {
        if (!this._isActionNode(event.target)) {
            this._stopPropagation(event);
        }
    },

    _isActionNode(node) {
        return Boolean(node && node.closest('.action'));
    },

    _off(event) {
        if (this._isActionNode(event.target)) {
            return;
        }

        const { onClose } = this.state;

        onClose();

        this._close();
    },

    _onButtonClick(event) {
        this._stopPropagation(event);

        const { onConfirm } = this.state;

        this._close();

        onConfirm();
    },

    _onClose(event) {
        this._stopPropagation(event);

        const { onClose } = this.state;

        onClose();

        this._close();
    },

    _close() {
        StatusBarActions.clear();
    },

    _stopPropagation(event) {
        event.stopPropagation();
    },

    render() {
        if (!this.state.isEnabled) {
            return (
                <div className="status-bar" ref="statusBar" />
            );
        }

        const { message, confirmText, rejectText, onConfirm, onReject } = this.state;

        return (
            <div className="status-bar status-bar_visible" ref="statusBar">
                <Icon type="selected" size="xl" />
                <span className="status-bar__message">
                    {message}
                </span>
                <div className="status-bar__actions">
                    <Button
                        cls="status-bar__action action"
                        text={confirmText}
                        onClick={onConfirm}
                        tone="grey"
                        view="default"
                        theme="action"
                        size="m"
                    />
                    {rejectText && onReject ?
                        <span className="status-bar__action_reject action" onClick={onReject}>{rejectText}</span> :
                        null}
                </div>
                <div className="status-bar__close status-bar__action action" onClick={this._onClose}>
                    <Icon type="light-cross" size="s" />
                </div>
            </div>
        );
    },
});

export default StatusBar;
