import PropTypes from 'prop-types';
import React from 'react';
import Select from 'react-bem-components/lib/Select';
import { Permissions } from 'constants/Common';
import { i18n, getTimezoneList } from 'lib/i18n';
import PermissionStore from 'stores/Permissions';

import Form from 'ui/Form';
import Input from 'ui/Input';
import DatePicker from 'ui/DatePicker';
import GenderPicker from 'ui/GenderPicker';

import AuthStore from 'stores/Auth';
import ConfigStore from 'stores/Config';

const DEFAULT_TIMEZONE = 'Europe/Moscow';
const DEFAULT_LANGUAGE = AuthStore.getLocale();

function isDisabled(user, key) {
    if (key === 'birthday') {
        return !PermissionStore.contains(Permissions.EDIT_USER_BIRTHDAY, user);
    }

    return !PermissionStore.contains(Permissions.EDIT_USER_INFO, user);
}

const Identity = React.createClass({

    _getTimezone(user) {
        if (user) {
            return user.get('timezone');
        }

        const currentUser = AuthStore.getUser();

        // при создании пользователя, делаем часовым поясом по умолчанию, пояс текущего пользователя
        return currentUser ? currentUser.get('timezone') : DEFAULT_TIMEZONE;
    },

    _getLanguage(user) {
        if (user) {
            return user.get('language');
        }

        const currentUser = AuthStore.getUser();

        return currentUser ? currentUser.get('language') : DEFAULT_LANGUAGE;
    },

    _getAcceptedLocales() {
        return ConfigStore.get('app.acceptedLocales')
            .toJS()
            .map(currentLang => ({
                val: currentLang,
                text: i18n(`common_vocab.lang.${currentLang}`),
            }));
    },

    render() {
        const { user, errors, locale } = this.props;
        const commonInputProps = {
            hasClear: true,
            size: 'm',
            autoComplete: false,
            width: 'available',
            disabled: isDisabled(user),
        };

        return (
            <div className="form__subsection user-form__identity">
                <Form.Item
                    key="lastName"
                    label={i18n('user.last_name')}
                    errors={errors.lastName}
                    required
                >
                    <Input
                        {...commonInputProps}
                        name={`name[last][${locale}]`}
                        val={user ? user.getLastName(locale) : null}
                    />
                </Form.Item>
                <Form.Item
                    key="firstName"
                    label={i18n('user.first_name')}
                    errors={errors.firstName}
                    required
                >
                    <Input
                        {...commonInputProps}
                        name={`name[first][${locale}]`}
                        val={user ? user.getFirstName(locale) : null}
                    />
                </Form.Item>
                <Form.Item
                    key="middleName"
                    label={i18n('user.middle_name')}
                    errors={errors.middleName}
                >
                    <Input
                        {...commonInputProps}
                        name={`name[middle][${locale}]`}
                        val={user ? user.getMiddleName(locale) : null}
                    />
                </Form.Item>
                <Form.Item
                    key="position"
                    label={i18n('user.position')}
                    errors={errors.position}
                >
                    <Input
                        {...commonInputProps}
                        name={`position[${locale}]`}
                        val={user ? user.getPosition(locale) : null}
                    />
                </Form.Item>
                <Form.Item
                    key="language"
                    label={i18n('user.language')}
                    errors={errors.language}
                >
                    <Select
                        name="language"
                        mode="radio"
                        options={this._getAcceptedLocales()}
                        val={this._getLanguage(user)}
                        disabled={isDisabled(user, 'language')}
                        width="available"
                        size="m"
                    />
                </Form.Item>
                <Form.Item
                    key="timezone"
                    label={i18n('user.timezone')}
                    errors={errors.timezone}
                >
                    <Select
                        name="timezone"
                        mode="radio"
                        options={getTimezoneList()}
                        val={this._getTimezone(user)}
                        disabled={isDisabled(user, 'timezone')}
                        width="available"
                        size="m"
                    />
                </Form.Item>
                <Form.Item
                    key="birthDate"
                    label={i18n('user.birth_date')}
                    errors={errors.birthday}
                >
                    <DatePicker
                        size="m"
                        name="birthday"
                        disabled={isDisabled(user, 'birthday')}
                        val={user ? user.getSerializedBirthDate() : null}
                    />
                </Form.Item>
                <Form.Item
                    key="gender"
                    label={i18n('user.gender')}
                    errors={errors.gender}
                >
                    <GenderPicker
                        size="m"
                        name="gender"
                        disabled={isDisabled(user, 'gender')}
                        val={user ? user.getGender() : null}
                    />
                </Form.Item>
            </div>
        );
    },
});

Identity.propTypes = {
    user: PropTypes.object,
    errors: PropTypes.object,
    locale: PropTypes.string,
};

Identity.defaultProps = {
    locale: 'ru',
    errors: {},
};

export default Identity;
