import React from 'react';
import { Spin, Link, Icon } from 'lego-on-react';
import block from 'bem-cn';
import GearIcon from 'components/Icon/Gear';
import BetaIcon from 'components/Icon/Beta';
import ServiceBadge from 'components2/ServiceBadge';
import metrika from 'api2/metrika';
import getService from 'lib2/getService';
import getServiceMainAction from 'lib2/getServiceMainAction';
import getCloudPromoUrl from 'lib2/getCloudPromoUrl';
import isCloudPromoService from 'lib2/isCloudPromoService';
import toggleService from 'lib2/toggleService';
import isAdmin from 'lib2/isAdmin';
import { i18n, pluralize } from 'i18n2';
import fetchResourceCount from './fetchResourceCount';
import './index.css';

const b = block('dashboard-card');

class DashboardCard extends React.PureComponent {
    constructor(props) {
        super(props);

        this.state = {
            pending: !getService(props.serviceSlug).ready,
        };

        this._onServiceToggle = this._onServiceToggle.bind(this);
        this._onCardClick = this._onCardClick.bind(this);
        this._onGearIconClick = this._onGearIconClick.bind(this);
        this._onActionLinkClick = this._onActionLinkClick.bind(this);
        this._onResourceCounterClick = this._onResourceCounterClick.bind(this);
    }

    componentDidMount() {
        let { serviceSlug } = this.props;

        if (getService(serviceSlug).with_resources) {
            fetchResourceCount(serviceSlug).then(resourceCount => {
                this.setState({ resourceCount });
            });
        }
    }

    _onServiceToggle() {
        let { serviceSlug } = this.props;

        toggleService({
            serviceSlug,
            targetStatus: true,
            onConfirm: () => this.setState({
                pending: true,
                modalOpen: false,
            }),
            onCancel: () => this.setState({
                modalOpen: false,
            }),
        })
            .then(output => {
                if (output && output.ModalComponent) {
                    this.setState({ ...output, modalOpen: true });
                }
            });

        metrika.send(
            'Подключить сервис',
            serviceSlug,
            window.location.pathname
        );
    }

    _onCardClick() {
        metrika.send(
            'Дашборд',
            'Клик по плитке',
            this.props.serviceSlug
        );
    }

    _onGearIconClick() {
        const { serviceSlug } = this.props;

        if (serviceSlug === 'mail') {
            metrika.sendExperiment(['pdb', 'dashboard'], isAdmin() ? 'Админ' : 'Не админ');
        }

        metrika.send(
            'Дашборд',
            'Клик по шестерёнке',
            serviceSlug
        );
    }

    _onActionLinkClick() {
        metrika.send(
            'Дашборд',
            'Клик по основному действию сервиса',
            this.props.serviceSlug
        );
    }

    _onResourceCounterClick() {
        metrika.send(
            'Дашборд',
            'Клик по ссылке на ресурсы сервиса',
            this.props.serviceSlug
        );
    }

    render() {
        let { serviceSlug } = this.props;
        let {
            pending,
            resourceCount,
            ModalComponent,
            modalProps,
            modalOpen,
        } = this.state;

        let service = getService(serviceSlug);
        let mainAction = getServiceMainAction(serviceSlug);
        let action;

        if (pending) {
            action = (
                <div className={b('action', { process: true })}>
                    <Spin
                        size="xxs"
                        progress
                    />
                    {i18n('dashboard.enabling')}
                </div>
            );
        } else if (service.can_be_enabled && !service.enabled) {
            action = (
                <Link
                    cls={b('action')}
                    theme="normal"
                    onClick={this._onServiceToggle}
                >
                    {i18n('dashboard.enable')}
                </Link>
            );
        } else if (mainAction.url && service.ready) {
            action = (
                <Link
                    cls={b('action')}
                    theme="normal"
                    url={mainAction.url}
                    onClick={this._onActionLinkClick}
                >
                    {i18n(`dashboard.card.${serviceSlug}.action`)}
                </Link>
            );
        }

        let { url, settings_url: settingsUrl } = service;
        let settingsTarget = undefined;

        // @see DIR-10291
        if (serviceSlug === 'mail' && !isAdmin()) {
            settingsUrl = null;
        }

        // @see DIR-9977 DIR-10400
        if (isCloudPromoService(service)) {
            settingsUrl = getCloudPromoUrl(service.slug);
            settingsTarget = '_blank';
            action = null;
        }

        return (
            <div
                className={b({
                    featured: service.featured,
                    'with-counter': Boolean(resourceCount),
                })}
                data-slug={serviceSlug}
            >
                <div className={b('frame')}>
                    <Link
                        cls={b('link')}
                        url={url}
                        theme="normal"
                        onClick={this._onCardClick}
                    />
                    {settingsUrl && (
                        <Link
                            url={settingsUrl}
                            cls={b('configure')}
                            theme="normal"
                            target={settingsTarget}
                            onClick={this._onGearIconClick}
                        >
                            <Icon cls={b('configure-icon')}>
                                <GearIcon />
                            </Icon>
                        </Link>
                    )}
                    <div className={b('header')}>
                        <Icon
                            cls={b('icon')}
                            url={service.icon}
                        />
                        <div className={b('title')}>
                            <span className={b('name')}>
                                {service.name}
                                {service.beta && <Icon cls={b('beta-icon')}><BetaIcon /></Icon>}
                            </span>
                            {service.with_subscriptions && (
                                <ServiceBadge
                                    serviceSlug={serviceSlug}
                                    cls={b('badge')}
                                    canShowTrialStatus
                                    canShowReadOnly={serviceSlug === 'tracker'}
                                    canShowPricing
                                />
                            )}
                        </div>
                    </div>
                    <div className={b('content')}>
                        {i18n(`dashboard.card.${serviceSlug}.description`)}
                    </div>
                    <div className={b('footer')}>
                        {Boolean(resourceCount) && (
                            <Link
                                theme="normal"
                                url={service.settings_url}
                                cls={b('counter')}
                                onClick={this._onResourceCounterClick}
                            >
                                {pluralize(
                                    resourceCount,
                                    i18n(`services.${serviceSlug}.counter`) || i18n('services.default.counter')
                                )}
                            </Link>
                        )}
                        {action}
                    </div>
                </div>
                {ModalComponent && (
                    <ModalComponent
                        {...modalProps}
                        visible={modalOpen}
                    />
                )}
            </div>
        );
    }
}

export default DashboardCard;
