import React from 'react';
import { Link } from 'lego-on-react';
import block from 'bem-cn';
import Modal from 'components2/Modal';
import hasPermission from 'lib2/hasPermission';
import notify from 'lib2/notify';
import directory from 'api2/directory';
import metrika from 'api2/metrika';
import { i18n } from 'i18n2';

const b = block('remove-domain-control');

export default class RemoveDomainControl extends React.PureComponent {
    constructor(props) {
        super(props);

        this.state = {};

        this._confirm = this._confirm.bind(this);
        this._cancel = this._cancel.bind(this);
        this._remove = this._remove.bind(this);
    }

    componentWillUnmount() {
        this._unmounted = true;
    }

    _confirm() {
        metrika.send(
            'Удалить домен',
            'Уверены?',
            this.props.trackId
        );

        this.setState({
            confirmOpen: true,
        });
    }

    _cancel() {
        metrika.send(
            'Удалить домен',
            'Отмена',
            this.props.trackId
        );

        this.setState({
            confirmOpen: false,
        });
    }

    _remove() {
        let { domain, onSubmit, trackId } = this.props;

        metrika.send(
            'Удалить домен',
            'Удалить',
            trackId
        );

        this.setState({ busy: true });

        directory
            .send('DELETE', `/v11/domains/${encodeURIComponent(domain.name)}/`)
            .then(({ ok, body }) => {
                metrika.send(
                    'Удалить домен',
                    ok ? 'Успешно' : 'Ошибка',
                    trackId
                );

                if (!ok) {
                    let message = i18n(`domain.error.remove.${body.code}`) ||
                        i18n('domain.error.remove.failed');

                    return notify(message, 'error');
                }

                notify(i18n('domain.status.removed'), 'success');

                if (onSubmit) {
                    return onSubmit();
                }
            })
            .finally(() => {
                if (!this._unmounted) {
                    this.setState({
                        busy: false,
                        confirmOpen: false,
                    });
                }
            });
    }

    render() {
        let { domain, cls, ...otherProps } = this.props;
        let { confirmOpen, busy } = this.state;

        if (domain.master || !hasPermission('remove_domain')) {
            return null;
        }

        return (
            <React.Fragment>
                <Link
                    cls={b({ busy }).mix(cls)}
                    theme="black"
                    pseudo
                    text={i18n('domain.action.remove')}
                    {...otherProps}
                    disabled={busy}
                    onClick={this._confirm}
                />
                <Modal
                    cls={b('confirm-modal')}
                    visible={confirmOpen}
                >
                    <Modal.Title>
                        {i18n('domain.remove.confirm.title')}
                    </Modal.Title>
                    <Modal.CloseButton onClick={this._cancel} />
                    <Modal.Body>
                        {i18n('domain.remove.confirm.message', { domain: domain.name })}
                    </Modal.Body>
                    <Modal.ConfirmActions
                        onConfirm={this._remove}
                        onCancel={this._cancel}
                        confirmCaption={i18n('domain.remove.confirm.positive')}
                    />
                    <Modal.Busy visible={busy} />
                </Modal>
            </React.Fragment>
        );
    }
}
