import React, { useState, useCallback } from 'react';
import { Button, Icon } from 'lego-on-react';
import { i18n, formatCurrency } from 'i18n2';

import getHelpUrl from 'lib2/getHelpUrl';

import { b } from './const';

import './BalanceHint.css';

const currency = 'RUB';

export const BalanceHint = React.memo((/** @type {IBalanceHintProps} */{
    debt,
    balance,
    monthPrice,
    summaryBalance,
    repaid,
}) => {
    const [open, setOpen] = useState(false);
    const onClick = useCallback(() => setOpen(!open), [open, setOpen]);

    // Если нет ни долга, ни лишних денег на балансе, то не показываем этот блок
    if (!summaryBalance && !balance && !repaid) {
        return null;
    }

    let hint = null;
    let view = null;

    if (debt > 0) {
        view = 'has-debt';
        hint = i18n('organization_remove.hint.text.debt'); // Пожалуйста, перед удалением организации погасите задолженность.
    } else {
        hint = i18n('organization_remove.hint.text.balance', {
            // href: 'https://balance.yandex.ru',
            href: getHelpUrl('ui.help.balance_main'),
        }); // Оплата за текущий месяц будет списана в начале следующего месяца. Остаток средств...
    }

    return (
        <div className={b('balance-hint', { open, view })}>
            <div
                className={b('balance-hint-title')}
                dangerouslySetInnerHTML={{ __html: open ?
                    i18n('organization_remove.hint.title.short') : // Баланс
                    i18n('organization_remove.hint.title.full', { // Баланс: 69₽
                        value: formatCurrency(summaryBalance, currency),
                        currency: '',
                    }),
                }}
            />
            {repaid && (
                <div className={b('balance-hint-repaid')}>
                    {i18n('organization_remove.hint.text.repaid') /* Задолженность погашена */}
                </div>
            )}
            <div
                className={b('balance-hint-text')}
                dangerouslySetInnerHTML={{ __html: hint }}
            />
            <table className={b('balance-hint-details')}>
                <tbody>
                    <tr>
                        <td>{i18n('organization_remove.hint.details.balance') /* Баланс организации */}</td>
                        <td dangerouslySetInnerHTML={{ __html: formatCurrency(balance, currency) }} />
                    </tr>
                    <tr>
                        <td>{i18n('organization_remove.hint.details.month_price') /* Сумма оплаты за текущий месяц: */}</td>
                        <td dangerouslySetInnerHTML={{ __html: formatCurrency(monthPrice, currency) }} />
                    </tr>
                    <tr className={b('balance-hint-summary')}>
                        <td>{i18n('organization_remove.hint.details.summary') /* Итоговая сумма: */}</td>
                        <td dangerouslySetInnerHTML={{ __html: formatCurrency(summaryBalance, currency) }} />
                    </tr>
                </tbody>
            </table>
            <Button
                text={i18n(open ?
                    'organization_remove.hint.details.hide' : // Скрыть подробности
                    'organization_remove.hint.details.show' // Показать подробности
                )}
                theme="normal"
                view="classic"
                tone="default"
                size="m"
                onClick={onClick}
                iconRight={(
                    <Icon
                        direction={open ? 'top' : 'bottom'}
                        glyph="type-arrow"
                    />
                )}
            />
        </div>
    );
});

/**
 * @typedef IBalanceHintProps
 * @property {Number} balance - текущий баланс организации
 * @property {Number} monthPrice - сумма потреблённых услгу в текущем месяце
 * @property {Number} debt - долг (ноль если нет; положительное число если долг есть)
 * @property {Number} summaryBalance - итоговый баланс (balance - monthPrice)
 * @property {Boolean} [repaid] - была ли погашена задолженность
 */
