import React from 'react';
import { Icon, Button } from 'lego-on-react';
import block from 'bem-cn';
import NavLink from 'components2/NavLink';
import CollapseIcon from 'components/Icon/Collapse';
import getCloudPromoUrl from 'lib2/getCloudPromoUrl';
import isCloudPromoService from 'lib2/isCloudPromoService';
import storage from 'lib2/storage';
import getServices from 'lib2/getServices';
import metrika from 'api2/metrika';
import { i18n } from 'i18n2';

import './index.css';

const b = block('service-sidebar');
const COLLAPSED_STATE_KEY = 'service_sidebar_collapsed';

export default class ServiceSidebar extends React.PureComponent {
    constructor(props) {
        super(props);

        this.state = {
            collapsed: storage.local.getItem(COLLAPSED_STATE_KEY) || false,
        };

        this._onCollapseClick = this._onCollapseClick.bind(this);
        this._onItemClick = this._onItemClick.bind(this);
    }

    _onCollapseClick() {
        let { collapsed } = this.state;

        this.setState({
            collapsed: !collapsed,
        });

        storage.local.setItem(COLLAPSED_STATE_KEY, !collapsed);

        metrika.send(
            'Сервисы',
            'Сайдбар',
            collapsed ? 'Развернуть сайдбар' : 'Свернуть сайдбар'
        );
    }

    _onItemClick(serviceSlug) {
        metrika.send(
            'Сервисы',
            'Сайдбар',
            'Выбор сервиса',
            serviceSlug
        );
    }

    render() {
        let { serviceSlug, cls } = this.props;
        let { collapsed } = this.state;

        return (
            <div className={b({ collapsed }).mix(cls)}>
                <div className={b('header')}>
                    <div className={b('title')}>
                        {i18n('service_page.sidebar.title')}
                    </div>
                    <div className={b('controls')}>
                        <Button
                            cls={b('collapse-button')}
                            theme="normal"
                            pin="circle-circle"
                            onClick={this._onCollapseClick}
                            icon={<Icon><CollapseIcon /></Icon>}
                        />
                    </div>
                </div>
                <div className={b('menu')}>
                    {getServices().map(item => {
                        // @see DIR-9977 DIR-10400
                        const isCloudService = Boolean(isCloudPromoService(item));

                        return item.available && (
                            <NavLink
                                theme="sidebar"
                                cls={b('menu-item', { active: item.slug === serviceSlug })}
                                url={isCloudService ? getCloudPromoUrl(item.slug) : item.settings_url}
                                onClick={() => this._onItemClick(item.slug)}
                                hard={isCloudService || item.slug === 'portal'}
                                target={isCloudService && '_blank'}
                                key={item.slug}
                            >
                                <span
                                    className={b('menu-icon')}
                                    title={item.name}
                                >
                                    <Icon url={item.icon} />
                                </span>
                                <span className={b('menu-title')}>
                                    {item.name}
                                </span>
                            </NavLink>
                        );
                    })}
                </div>
            </div>
        );
    }
}
