import _ from 'lodash';

import AuthStore from 'stores/Auth';
import ServiceStore from 'stores/Services';
import PermissionStore from 'stores/Permissions';
import ConfigStore from 'stores/Config';

import Url from 'lib/Url';
import { i18n } from 'lib/i18n';

const CONFIGURABLE_SERVICES = ['wiki', 'tracker', 'forms', 'staff', 'calendar', 'mail'];
const ADMIN_SERVICES = ['portal'];
const SERVICES_AVAILABLE_WITH_OWNED_DOMAIN = ['mail', 'calendar'];

function getProp(item, propName) {
    return typeof item[propName] === 'function' ? item[propName]() : item[propName];
}

export const Service = {

    isAvailable(id) {
        // TODO хорошо бы перенести эту проверку в ноду
        if (id === 'webmaster') {
            return AuthStore.isAdmin();
        }

        const isEnabled = this.isEnabled(id);

        if (isEnabled) {
            // Если сервис вклчючен, то показываем его
            return true;
        }

        if (ADMIN_SERVICES.indexOf(id) !== -1) {
            return PermissionStore.allowsOrganizationStructureEditing() || AuthStore.isAdmin();
        }

        // Если сервис выключен, то показываем только сервисы которые которые можно включить
        return this.isConfigurableService(id);
    },

    isConfigurableService(service) {
        // только админу организации(если есть прово на управление сервисами)
        if (CONFIGURABLE_SERVICES.indexOf(service) === -1 || !PermissionStore.allowsManageServices()) {
            return false;
        }

        // Показываем только сервисы которые можно включить с подтвержденным доменом
        return AuthStore.hasOwnedDomains() || SERVICES_AVAILABLE_WITH_OWNED_DOMAIN.indexOf(service) === -1;
    },

    isEnabled(id) {
        return AuthStore.isServiceEnabled(id);
    },

    getList(items) {
        return items.map(item => this._preprocessItem(item));
    },

    _preprocessItem(item) {
        const isService = item.id.indexOf('service.') === 0;
        const itemProps = _.extend({}, item, {
            url: getProp(item, 'url'),
            weight: getProp(item, 'weight'),
            available: getProp(item, 'available'),
            settingsLink: getProp(item, 'settingsLink'),
            actionLink: getProp(item, 'actionLink'),
            isService,
        });
        const actionLinkParams = {
            nickname: AuthStore.getUser().getNickname(),
            org_id: AuthStore.getOrganizationId(),
            uid: AuthStore.getUserId(),
        };

        if (isService) {
            itemProps.id = item.id.replace('service.', '');
        }

        itemProps.ready = AuthStore.isServiceReady(itemProps.id);
        itemProps.text = i18n(`dashboard.card.${itemProps.id}.description`);
        itemProps.actionLink = itemProps.actionLink ||
            Url.build(ConfigStore.get(`ui.serviceActions.${itemProps.id}`) || '', actionLinkParams);

        itemProps.action = i18n(`dashboard.card.${itemProps.id}.action`);
        itemProps.title = itemProps.title || i18n(`dashboard.card.${itemProps.id}.title`);

        if (isService) {
            itemProps.active = AuthStore.isServiceEnabled(itemProps.id);
            itemProps.settingsLink = itemProps.settingsLink || Url.getPath('services', itemProps.id);

            itemProps.title = ServiceStore.getName(itemProps.id) || itemProps.title;
            itemProps.available &= this.isAvailable(itemProps.id);
        }

        itemProps.url = itemProps.url || ConfigStore.getRelatedServiceUrl(itemProps.id);

        return itemProps;
    },

};

window.ya.connect.services = { Service };
