import Immutable from 'immutable';
import _ from 'lodash';

import UserStore from 'stores/Users';

const DEFAULTS = {
    id: undefined,
    name: undefined,
    label: undefined,
    organization_type: undefined,
    vip: undefined,
    partner_id: undefined,
    admin: undefined,
    owner_login: undefined,
    domains: undefined,
    services: undefined,
    has_debt: undefined,
    logo: undefined,
    header: undefined,

    head_id: undefined,
    phone_number: undefined,
    email: undefined,
    fax: undefined,
    language: undefined,
    country: undefined,

    subscription: undefined,
    subscription_plan: undefined,
    subscription_plan_expires_at: undefined,
    has_natural_domains: undefined,
    has_owned_domains: undefined,
    can_users_change_password: undefined,
    is_blocked: undefined,

    user_count: undefined,
    department_count: undefined,
    group_count: undefined,
    admin_count: undefined,

    root_departments: undefined,
};

const CONTACTS = [
    'phone_number', 'email', 'fax',
];

const LEGAL_INFO = [
    'law_address', 'real_address',
    'ogrn', 'inn', 'trc', 'account', 'corr_acc',
];

class Organization extends Immutable.Record(DEFAULTS) {
    static PRIMARY_KEY = 'id';
    static TYPE_NAME = 'organization';

    static create(data) {
        const nextData = _.pick(data, Object.keys(DEFAULTS));

        return new Organization(Immutable.fromJS(nextData));
    }

    getId() {
        return this.get(Organization.PRIMARY_KEY);
    }

    getType() {
        return Organization.TYPE_NAME;
    }

    equals(x) {
        return (x instanceof Organization) && x.getId() === this.getId();
    }

    getLabel() {
        return this.get('label');
    }

    getName() {
        return this.get('name');
    }

    getLanguage() {
        return this.get('language');
    }

    getAdmin() {
        return UserStore.get(this.get('admin'));
    }

    getAdminId() {
        return this.get('admin');
    }

    getContacts() {
        return CONTACTS.map(type => ({ type, value: this.get(type) }));
    }

    getRootDepartments() {
        const departments = this.get('root_departments');

        return departments ? departments.toJS() : [];
    }

    getLegalInfo() {
        return LEGAL_INFO.map(type => ({ type, value: this.get(type) }));
    }

    hasLegalInfo() {
        for (let i = 0, n = LEGAL_INFO.length; i < n; i++) {
            if (this.get(LEGAL_INFO[i]) !== undefined) {
                return true;
            }
        }

        return false;
    }

    hasOwnedDomains() {
        return this.get('has_owned_domains');
    }

    getHead() {
        return UserStore.get(String(this.get('head_id')));
    }

    getDomain() {
        return this.getIn(['domains', 'master']);
    }

    getServices() {
        const services = this.get('services');

        return services ? services.toJS() : [];
    }

    getLogoUrl(size = 'orig') {
        const url = this.getIn(['logo', size, 'url']);

        if (url) {
            return url.replace(/^http:/, 'https:');
        }
    }

    hasDebt() {
        return this.get('has_debt');
    }

    isVip() {
        // vip - массив причин быть vip-организацией
        // если массив непустой, организация считается вип
        const vip = this.get('vip');

        return Boolean(vip && vip.size);
    }
}

export default Organization;
