import _ from 'lodash';
import AuthStore from 'stores/Auth';
import UserStore from 'stores/Users';
import DepartmentStore from 'stores/Departments';
import GroupStore from 'stores/Groups';
import User from 'records/User';
import Department from 'records/Department';
import Group from 'records/Group';
import Service from 'records/Service';

import SearchItem from 'records/SearchItem';

const STORE_DATA_MAP = {
    user: 'users',
    department: 'departments',
    group: 'groups',
};

export function getName(item, locale) {
    if (!item || !item.name) {
        return;
    }

    if (!locale) {
        locale = AuthStore.getLocale();
    }

    const { name } = item;

    if (typeof name === 'object') {
        return name[locale];
    }

    if (typeof name === 'string') {
        return name;
    }
}

export function toRecord(item) {
    const id = String(item.id || item.object.id);
    const type = item.object_type || item.type;

    if (_.has(item, 'object.search_item') || _.has(item, 'search_item')) {
        return SearchItem.create(item.object || item);
    }

    switch (type) {
        case 'user':
        case 'users':
            return UserStore.get(id) || User.create(item.object || item);
        case 'department':
        case 'departments':
            return DepartmentStore.get(id) || Department.create(item.object || item);
        case 'group':
        case 'groups':
            return GroupStore.get(id) || Group.create(item.object || item);
        case 'service':
            return Service.create(item.object || item);
    }
}

export function toCompactObject(item) {
    if (item === null || item === undefined) {
        return item;
    }

    if (typeof item !== 'object') {
        return null;
    }

    if (item.getId && item.getType) {
        return {
            id: Number(item.getId()),
            type: item.getType(),
        };
    } else if (item.id && item.type) {
        return {
            id: Number(item.id),
            type: item.type,
        };
    } else if (item.object) {
        return {
            id: Number(item.id || item.object.id),
            type: item.type || item.object_type,
        };
    }
}

export function serialize(item) {
    if (!item || !item.toJS || !item.getType) {
        return item;
    }

    return { type: item.getType(), object: item.toJS() };
}

export function findIndex(item, list) {
    if (!list || !list.length) {
        return -1;
    }

    let index = -1;

    list.forEach((listItem, listIndex) => {
        if (item && item.equals(listItem)) {
            index = listIndex;
        }
    });

    return index;
}

export function toStoreData(items) {
    const storeData = {
        users: {},
        departments: {},
        groups: {},
    };

    (items || []).forEach(item => {
        let storableItem = {};

        if (item.getId && item.getType && item.toJS) {
            storableItem = {
                id: item.getId(),
                type: item.getType(),
                object: item.toJS(),
            };
        } else if (item.object) {
            storableItem = {
                id: item.id || item.object.id,
                type: item.type || item.object_type,
                object: item.object,
            };
        }

        const storeType = STORE_DATA_MAP[storableItem.type];

        if (storeType) {
            storeData[storeType][storableItem.id] = storableItem.object;
        }
    });

    return storeData;
}
