/**
 * Хранилище данных из конфига
 */

import _ from 'lodash';
import Immutable from 'immutable';
import { ReduceStore } from 'flux/utils';
import { ActionTypes } from 'constants/Common';
import AuthStore from 'stores/Auth';
import dispatcher from 'dispatcher';
import Url from 'lib/Url';
import template from 'lib/template';

class ConfigStore extends ReduceStore {
    getInitialState() {
        return new Immutable.Map();
    }

    reduce(state, action) {
        const { data, type } = action;

        switch (type) {
            case ActionTypes.RECEIVE_CONFIG_DATA:
                return state.merge(data);
        }

        return state;
    }

    /**
     * Возвращает значение по ключу из состояния хранилища
     * @method  get
     * @param   {String}  key
     * @param   {Object}  [data]
     * @returns {String}
     */
    get(key, data) {
        let value;

        if (this._state.isEmpty()) {
            value = _.get(window.ya, `connect.config.${key}`);
        } else {
            value = this._state.getIn(key.split('.'));
        }

        if (value && data) {
            return template.build(value, data);
        }

        return value;
    }

    /**
     * Возвращает url хелпа с параметрами для встроенных в страницу хелпа форм
     * @param   {String} key
     * @param   {Object?} keyData
     * @returns {String}
     * @see Url.setFormOptions(url, formData, formId)
     */
    getHelpUrl(key, keyData) {
        const org = AuthStore.getOrganization();
        let basicFormData = {};

        if (org) {
            basicFormData = {
                org_id: org.getId(),
                v: org.isVip() ? 1 : 0,
            };
        }

        return Url.setFormOptions(this.get(key, keyData), basicFormData);
    }

    /**
     * Возвращает url по ключу
     * @method  getUrl
     * @param   {String}  key
     * @returns {String}
     */
    getUrl(key) {
        const mappedUrl = this.get(key);

        if (mappedUrl) {
            const session = AuthStore.getSession();

            return Url.build(mappedUrl, {
                uid: session ? session.getId() : null,
            });
        }
    }

    /**
     * Возвращает url по ключу из набора relatedServices
     * @method  getRelatedServiceUrl
     * @param   {String}  key
     * @returns {String}
     */
    getRelatedServiceUrl(key) {
        return this.getUrl(`ui.relatedServices.${key}`);
    }

    /**
     * Возвращает url по слагу сервиса из набора ui.help.bind.faq или отдает дефолтный
     * @method  getServiceBindFaqUrl
     * @param   {String} serviceSlug
     * @returns {String}
     */
    getServiceBindFaqUrl(serviceSlug) {
        return this.getUrl(`ui.help.bind.faq.${serviceSlug}`) || this.getUrl('ui.help.bind.faq.default');
    }

    /**
     * Возвращает url по слагу сервиса из набора ui.help.bind.advantages или отдает дефолтный
     * @method  getServiceBindAdvantagesUrl
     * @param   {String} serviceSlug
     * @returns {String}
     */
    getServiceBindAdvantagesUrl(serviceSlug) {
        return this.getUrl(`ui.help.bind.advantages.${serviceSlug}`) ||
            this.getUrl('ui.help.bind.advantages.default');
    }

    /**
     * Возвращает url по слагу сервиса из набора ui.help.bind.more или отдает дефолтный
     * @method  getServiceBindMoreUrl
     * @param   {String} serviceSlug
     * @returns {String}
     */
    getServiceBindMoreUrl(serviceSlug) {
        return this.getUrl(`ui.help.bind.advantages.${serviceSlug}`) ||
            this.getUrl('ui.help.bind.advantages.default');
    }

    /**
     * Возвращает url по слагу сервиса из набора ui.help.bind.subscribe или отдает дефолтный
     * @method  getServiceBindSubscribeUrl
     * @param   {String} serviceSlug
     * @returns {String}
     */
    getServiceBindSubscribeUrl(serviceSlug) {
        return this.getUrl(`ui.help.bind.subscribe.${serviceSlug}`) ||
            this.getUrl('ui.help.bind.subscribe.default');
    }

    getRelatedApps(serviceId) {
        const apps = this.get(`ui.relatedApps.${serviceId}`);

        return apps ? apps.toJS() : null;
    }

    getOrganizationSignUpUrl() {
        return Url.build(this.get('passport.org.registration'), {
            retpath: location.href,
            tld: this.get('app.tld'),
        });
    }
}

export default window.ya.connect.ConfigStore = new ConfigStore(dispatcher);
