/**
 * Хранилище данных о доменах организации
 */

import _ from 'lodash';
import Immutable from 'immutable';
import { ReduceStore } from 'flux/utils';
import { ActionTypes } from 'constants/Common';
import dispatcher from 'dispatcher';
import Domain from 'records/Domain';

class DomainStore extends ReduceStore {
    getInitialState() {
        return new Immutable.Map();
    }

    reduce(state, action) {
        const { data, type } = action;

        switch (type) {
            case ActionTypes.RECEIVE_MIXED_DATA:
                return this._receiveMixedData(state, data.domains);
        }

        return state;
    }

    _receiveMixedData(state, domainData) {
        if (!domainData) {
            return state;
        }

        return new Immutable.Map(domainData)
            .map(domain => Domain.create(domain));
    }

    /**
     * Возвращает значение по ключу из состояния хранилища
     * @method  get
     * @param   {String}  key
     * @returns {String}
     */
    get(key) {
        return this.getState().get(key);
    }

    has(key) {
        return this._state.has(key);
    }

    /**
     * Получение списка подтвержденных доменов
     * @returns {Array}
     */
    getConfirmedDomains() {
        const domains = this.getState().toJS();

        return _.chain(domains)
            .values()
            .filter('owned')
            .value();
    }

    /**
     * Получение домена по имени
     * @param {String} domainName
     * @returns {Domain}
     */
    getDomain(domainName) {
        if (!domainName) {
            return null;
        }

        try {
            domainName = decodeURIComponent(domainName);
        } catch (e) {
            // empty
        }

        return this.getState().get(domainName);
    }

    /**
     * Получение списка доменов
     * @returns {Array}
     */
    getAll() {
        return this.getState().toArray();
    }

    getCount() {
        return this.getState().size;
    }

    /**
     * Возвращает список делегированных доменов
     * @returns {Array}
     */
    getOwnedDomains() {
        return this.getState()
            .toArray()
            .filter(domain => domain.get('delegated') || domain.get('owned'))
            .map(domain => domain.get('name'));
    }

    /**
     * Есть ли в организации делегированные домены
     * @returns {Boolean}
     */
    hasOwnedDomains() {
        const domains = this.getOwnedDomains();

        return Boolean(domains && domains.length);
    }

    /**
     * Получение основного домена
     * @returns {Object}
     */
    getMasterDomain() {
        const masterDomain = this.getState().find(domain => domain.get('master'));

        return masterDomain && masterDomain.get('name');
    }

    hasNonTechnicalDomains() {
        const list = this.getAll();

        if (!list || list.length === 0) {
            return false;
        }

        return list.filter(domain => !domain.get('tech')).length !== 0;
    }
}

export default window.ya.connect.DomainStore = new DomainStore(dispatcher);
