/**
 * Хранилище данных о настройках организации
 */

import _ from 'lodash';
import Immutable from 'immutable';
import { ReduceStore } from 'flux/utils';
import { ActionTypes } from 'constants/Common';
import dispatcher from 'dispatcher';
import Dkim from 'records/Dkim';

class OrgSettingsStore extends ReduceStore {
    getInitialState() {
        return new Immutable.Map();
    }

    reduce(state, action) {
        const { data, type } = action;

        switch (type) {
            case ActionTypes.RECEIVE_MIXED_DATA:
            case ActionTypes.RECEIVE_ORGANIZATION_SETTINGS:
                return this._receiveMixedData(state, data.organizationSettings || {});
        }

        return state;
    }

    _receiveMixedData(state, data) {
        if (!data.dkim) {
            return state.merge(data);
        }

        const dkim = Immutable.Map(data.dkim)
            .map(item => Dkim.create(item));

        return state.merge(_.extend({}, data, { dkim }));
    }

    /**
     * Есть ли данные о настройках организации
     * @returns {Boolean}
     */
    isEmpty() {
        return this.getState().isEmpty();
    }

    /**
     * Возвращает массив статусов DKIM для каждого домена
     * @returns {Array}
     */
    getDkimList() {
        const dkim = this.get('dkim');

        return dkim ? dkim.toArray() : [];
    }

    /**
     * Получение данных о DKIM по имени домена
     * @param {String} domain
     * @returns {Dkim|null}
     */
    getDkimByDomain(domain) {
        const dkim = this.get('dkim');

        return dkim ? dkim.get(domain) : null;
    }

    /**
     * Возвращает состояния dkim
     * @returns {Boolean}
     */
    isDkimEnabled() {
        const dkim = this.getDkimList();

        if (!dkim.length) {
            return false;
        }

        return dkim.reduce((prev, current) => prev && current.get('enabled') === 'yes', true);
    }

    /**
     * Возвращает значение по ключу из состояния хранилища
     * @method  get
     * @param   {String}  key
     * @returns {String}
     */
    get(key) {
        return this.getState().getIn(key.split('.'));
    }
}

export default window.ya.connect.OrgSettingsStore = new OrgSettingsStore(dispatcher);
