import PropTypes from 'prop-types';
import React from 'react';
import Select from 'react-bem-components/lib/Select';
import { i18n } from 'lib/i18n';
import Input from 'ui/Input';

import './index.css';

const MONTHS = [
    'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
    'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec',
];

const MONTH_LIST = ['month'].concat(MONTHS)
    .map((item, index) => ({
        text: i18n(`common_vocab.calendar.${item}`),
        val: index === 0 ? '' : String(index),
    }));

function toNumber(x) {
    return x === undefined || x === '' ? NaN : Number(x);
}

// '1900-12-31' > [ 1900, 12, 31 ]
function toValues(value) {
    const components = (value || '').split('-')
        .map(item => {
            const parsedItem = toNumber(item);

            return isNaN(parsedItem) ? '' : String(parsedItem);
        });

    return { year: components[0], month: components[1], date: components[2] };
}

// [ 1900, 12, 31 ] > '1900-12-31'
function toValue(values) {
    const components = (values ? [values.year, values.month, values.date] : []).filter(Boolean);

    if (components.length === 0) {
        return '';
    }

    return components
        .map(item => item === undefined ? '' : item)
        .map(item => (item && item.length < 2 ? '0' : '') + item)
        .join('-');
}

const DatePicker = React.createClass({

    getInitialState() {
        return { values: toValues(this.props.val) };
    },

    focus() {
        return this.refs.date.focus();
    },

    select() {
        return this.refs.date.select();
    },

    value() {
        return toValue([
            this.refs.year.value(),
            this.refs.month.value(),
            this.refs.date.value(),
        ]);
    },

    _handleDateChange(payload) {
        payload.type = 'date';
        this._handleChange(payload);
    },

    _handleMonthChange(payload) {
        payload.type = 'month';
        this._handleChange(payload);
    },

    _handleYearChange(payload) {
        payload.type = 'year';
        this._handleChange(payload);
    },

    _handleChange(payload) {
        if (payload.type !== undefined) {
            const values = this.state.values;

            values[payload.type] = payload.value;
            this.setState({ values });
        }
    },

    render() {
        const { size, name, disabled } = this.props;
        const { values } = this.state;

        return (
            <div className="date-picker">
                <input type="hidden" name={name} value={toValue(values)} />
                <div className="date-picker__date">
                    <Input
                        ref="date"
                        size={size}
                        disabled={disabled}
                        placeholder={i18n('common_vocab.calendar.date')}
                        val={values.date}
                        onChange={this._handleDateChange}
                    />
                </div>
                <div className="date-picker__month">
                    <Select
                        ref="month"
                        options={MONTH_LIST}
                        size={size}
                        disabled={disabled}
                        val={values.month}
                        onChange={this._handleMonthChange}
                    />
                </div>
                <div className="date-picker__year">
                    <Input
                        ref="year"
                        size={size}
                        disabled={disabled}
                        placeholder={i18n('common_vocab.calendar.year')}
                        val={values.year}
                        onChange={this._handleYearChange}
                    />
                </div>
            </div>
        );
    },

});

// @if NODE_ENV='development'
DatePicker.propTypes = {
    val: PropTypes.any,
    size: PropTypes.string,
    name: PropTypes.string,
    disabled: PropTypes.bool,
};
// @endif

export default DatePicker;
