/**
 * Валидация данных пользователя, полученных при редактировании/добавлении DNS
 */

import _ from 'lodash';
import Immutable from 'immutable';
import { i18n } from 'lib/i18n';

const REGEXP_SPACE = /\s/;

const EMPTY_VALUE_ERROR = i18n('dns_validation.empty');

const fieldsValidators = {
    content(value) {
        if (!value) {
            return EMPTY_VALUE_ERROR;
        }
    },

    target(value, type) {
        if (!value && type === 'srv') {
            return EMPTY_VALUE_ERROR;
        }
    },

    subdomain(value, type) {
        if (!value && type === 'srv') {
            return EMPTY_VALUE_ERROR;
        }
    },

    priority(value, type) {
        if (!value && ['mx', 'srv'].indexOf(type) !== -1) {
            return EMPTY_VALUE_ERROR;
        }
    },

    weight(value, type) {
        if (!value && type === 'srv') {
            return EMPTY_VALUE_ERROR;
        }
    },

    port(value, type) {
        if (!value && type === 'srv') {
            return EMPTY_VALUE_ERROR;
        }
    },

    refresh(value, type) {
        if (!value && type === 'soa') {
            return EMPTY_VALUE_ERROR;
        }
    },

    retry(value, type) {
        if (!value && type === 'soa') {
            return EMPTY_VALUE_ERROR;
        }
    },

    expire(value, type) {
        if (!value && type === 'soa') {
            return EMPTY_VALUE_ERROR;
        }
    },

    neg_cache(value, type) {
        if (!value && type === 'soa') {
            return EMPTY_VALUE_ERROR;
        }
    },

    admin_mail(value, type) {
        if (!value && type === 'soa') {
            return EMPTY_VALUE_ERROR;
        }

        if (REGEXP_SPACE.test(value.trim())) {
            return i18n('dns_validation.wrong_input');
        }
    },
};

function getErrors(data = {}) {
    const errors = {};
    const type = data.type ? data.type.toLowerCase() : '';

    if (_.isEmpty(data)) {
        return new Immutable.Map(errors);
    }

    _.forOwn(data, (value, key) => {
        const validator = fieldsValidators[key];
        const error = _.isFunction(validator) ? validator(value, type) : null;

        if (error) {
            errors[key] = error;
        }
    });

    return new Immutable.Map(errors);
}

export default {
    getErrors,
};
