import validator from 'validator';
import { i18n } from 'lib/i18n';
import getErrorMap from './getErrorMap';

const maxLength = {
    LONG_NAME: 512,
    PHONE: 256,
    EMAIL: 256,
    POSTAL_CODE: 64,
    POSTAL_ADDRESS: 512,
    LEGAL_ADDRESS: 512,
    INN: 64,
    KPP: 64,
    BIK: 64,
    ACCOUNT: 20,
};

const legalPersonValidator = {

    getLongNameError(data) {
        const { long_name: longName } = data;

        if (!longName) {
            return i18n('subscription_validation.legal_person.long_name_undefined');
        }

        if (longName.length > maxLength.LONG_NAME) {
            return i18n('subscription_validation.legal_person.long_name_too_long', {
                maxLength: maxLength.LONG_NAME,
            });
        }

        return null;
    },

    getPhoneError(data) {
        const { phone } = data;

        if (!phone) {
            return i18n('subscription_validation.common.phone_undefined');
        }

        if (phone.length > maxLength.PHONE) {
            return i18n('subscription_validation.common.phone_too_long', {
                maxLength: maxLength.PHONE,
            });
        }

        return null;
    },

    getEmailError(data) {
        const { email } = data;

        if (!email) {
            return i18n('subscription_validation.common.email_undefined');
        }

        if (!validator.isEmail(email)) {
            return i18n('subscription_validation.common.invalid_email');
        }

        if (email.length > maxLength.EMAIL) {
            return i18n('subscription_validation.common.email_too_long', {
                maxLength: maxLength.EMAIL,
            });
        }

        return null;
    },

    getPostalCodeError(data) {
        const { postal_code: postalCode } = data;

        if (!postalCode) {
            return i18n('subscription_validation.legal_person.postal_code_undefined');
        }

        if (postalCode.length > maxLength.POSTAL_CODE) {
            return i18n('subscription_validation.legal_person.postal_code_too_long', {
                maxLength: maxLength.POSTAL_CODE,
            });
        }

        return null;
    },

    getPostalAddressError(data) {
        const { postal_address: postalAddress } = data;

        if (!postalAddress) {
            return i18n('subscription_validation.legal_person.postal_address_undefined');
        }

        if (postalAddress.length > maxLength.POSTAL_ADDRESS) {
            return i18n('subscription_validation.legal_person.postal_address_too_long', {
                maxLength: maxLength.POSTAL_ADDRESS,
            });
        }

        return null;
    },

    getLegalAddressError(data) {
        const { legal_address: legalAddress } = data;

        if (!legalAddress) {
            return i18n('subscription_validation.legal_person.legal_address_undefined');
        }

        if (legalAddress.length > maxLength.LEGAL_ADDRESS) {
            return i18n('subscription_validation.legal_person.legal_address_too_long', {
                maxLength: maxLength.LEGAL_ADDRESS,
            });
        }

        return null;
    },

    getInnError(data) {
        const { inn } = data;

        if (!inn) {
            return i18n('subscription_validation.legal_person.inn_undefined');
        }

        if (inn.length > maxLength.INN) {
            return i18n('subscription_validation.legal_person.inn_too_long', {
                maxLength: maxLength.INN,
            });
        }

        return null;
    },

    getKppError(data) {
        const { kpp } = data;

        if (kpp && kpp.length > maxLength.KPP) {
            return i18n('subscription_validation.legal_person.kpp_too_long', {
                maxLength: maxLength.KPP,
            });
        }

        return null;
    },

    getBikError(data) {
        const { bik } = data;

        if (!bik) {
            return i18n('subscription_validation.legal_person.bik_undefined');
        }

        if (bik.length > maxLength.BIK) {
            return i18n('subscription_validation.legal_person.bik_too_long', {
                maxLength: maxLength.BIK,
            });
        }

        return null;
    },

    getAccountError(data) {
        const { account } = data;

        if (!account) {
            return i18n('subscription_validation.legal_person.account_undefined');
        }

        if (account.length > maxLength.ACCOUNT) {
            return i18n('subscription_validation.legal_person.account_too_long', {
                maxLength: maxLength.ACCOUNT,
            });
        }

        return null;
    },

};

legalPersonValidator.getErrors = getErrorMap(legalPersonValidator);

export default legalPersonValidator;
