#!/bin/bash

set -e
# set -x

BASE_ENVIRONMENT="test"
BASE_DOMAIN=portal.test.ws.yandex.ru

PORTAL_VERSION="$(git rev-parse --abbrev-ref HEAD)"
GIT_REPOSITORY="$(git config --get remote.origin.url)"

# create is default action
CREATE_ACTION_NAME="create"
UPDATE_ACTION_NAME="update"
DELETE_ACTION_NAME="delete"

ACTION=$CREATE_ACTION_NAME

while [[ $# -gt 0 ]]
do
    key="$1"

    case $key in
        -v|--version)
        PORTAL_VERSION="$2"
        shift # past argument
        shift # past value
        ;;
        --repository)
        GIT_REPOSITORY=$2
        shift # past argument
        shift # past value
        ;;
        -c|--create)
        ACTION=$CREATE_ACTION_NAME
        shift # past argument
        ;;
        -u|--update)
        ACTION=$UPDATE_ACTION_NAME
        shift # past argument
        ;;
        -d|--delete)
        ACTION=$DELETE_ACTION_NAME
        shift # past argument
        ;;
    esac
done

FEATURE_ENVIRONMENT="$(echo "${PORTAL_VERSION}" | tr '[:upper:]' '[:lower:]')"

# при обновлении берем за оснуву ранее созданное окружение
if [ $ACTION = $UPDATE_ACTION_NAME ]; then
    BASE_ENVIRONMENT=$FEATURE_ENVIRONMENT
fi

build-package() {
    CONNECT_APP__ASSETS="/portal/static" CONNECT_APP__FREEZE_PATH="/portal/_" \
        npm run build -- --feature --local -v $PORTAL_VERSION --repository $GIT_REPOSITORY;
    echo "Собран и запушен пакет c версией $PORTAL_VERSION";
}

dump-environment() {
    releaser env-dump \
        --project workspace \
        --application portal \
        --environment $BASE_ENVIRONMENT \
        --dump portal-test.json

    # Отключим webauth, так как в эти стенды нам не надо пускать ассесоров
    cat "portal-test.json" | sed 's/webAuthCheckIdm: true/webAuthCheckIdm: false/' > tmp.json
    mv tmp.json "portal-test.json"
}

deploy-environment() {
    releaser deploy \
        --image registry.yandex.net/workspace/portal \
        --version "$PORTAL_VERSION" \
        --components 'back(IVA=1)' \
        --project workspace \
        --applications portal \
        --environment "$FEATURE_ENVIRONMENT" \
        --dump portal-test.json \
        --var "CONNECT_APP__ASSETS=/portal/static" \
        --var "CONNECT_APP__FREEZE_PATH=/portal/_" \
        --var "LOCAL_STATIC=true"
}

add-domain() {
    releaser add-domain \
        --project workspace \
        --application portal \
        --environment "$FEATURE_ENVIRONMENT" \
        --domain "$FEATURE_ENVIRONMENT.$BASE_DOMAIN" \
        --domain-type connect-internal-test

    echo "В окружени workspace.portal.$FEATURE_ENVIRONMENT добавлен домен $FEATURE_ENVIRONMENT.$BASE_DOMAIN"
}

delete-environment() {
    releaser env-delete \
        --project workspace \
        --application portal \
        --environment "$FEATURE_ENVIRONMENT"
}

if [ $ACTION = $UPDATE_ACTION_NAME ];then
    build-package;
    dump-environment;
    deploy-environment;

elif [ $ACTION = $DELETE_ACTION_NAME ];then
    delete-environment;

else
    build-package;
    dump-environment;
    deploy-environment;
    add-domain;
fi

rm portal-test.json

