/**
 * Мидлваре который проверяет статус сервиса на который пытыается зайти пользователя
 * и в случае когда сервис отключен редиректит пользователя на страницу /forbidden
 */

const url = require('url');
const _ = require('lodash');
const logger = require('../lib/logger');
const responseContext = require('../lib/responseContext');

const checkedSlugs = [
    'staff',
    'dashboard',
];

const regexpReplacePrefix = new RegExp('^\/portal', 'gi');

/**
 * Генерирует словарь в котором ключами являются пути к разделу с сервисом,
 * а значениями название сервиса
 * @param {Request} req
 * @returns {Object}
 */
function generateServicePathsDictionary(req) {
    return Object.keys(req.context.services).reduce((result, serviceName) => {
        if (checkedSlugs.indexOf(serviceName) === -1) {
            return result;
        }

        const serviceUrl = _.get(req.context.services, [serviceName, 'url'], '');
        const parsedUrl = url.parse(serviceUrl, false, true);
        let servicePath = _.get(parsedUrl, 'path', '');

        // хак для дев окружения
        if (req.config.app.env === 'development') {
            servicePath = servicePath.replace(regexpReplacePrefix, '');
        }

        if (servicePath) {
            result[servicePath] = serviceName;
        }

        return result;
    }, {});
}

/**
 * Возвращает название сервиса в коннекте в котором находится пользователь
 * @param {Request} req
 * @returns {String|null}
 */
function getServiceName(req) {
    if (!req.context.services) {
        return null;
    }

    const currentPath = req.originalUrl || '';
    let matchedServiceName = null;
    const dict = generateServicePathsDictionary(req);

    Object.keys(dict).forEach(servicePath => {
        if (currentPath.indexOf(servicePath) === 0) {
            matchedServiceName = dict[servicePath];
        }
    });

    return matchedServiceName;
}

module.exports = function(req, res, next) {
    const serviceName = getServiceName(req);

    if (!serviceName) {
        return next();
    }

    const orgId = req.context.state.organization;
    const services = _.get(req.context.organizations, [orgId, 'services']);

    const service = _.find(services, ['slug', serviceName]);
    const isEnabled = _.get(service, 'enabled', false);
    const isServiceActive = _.get(service, 'ready', false) ? isEnabled : !isEnabled;

    if (!service || !isServiceActive) {
        const originalUrl = responseContext.getUrl(req, req.originalUrl);
        const redirectUrl = responseContext.getUrl(req,
            `/forbidden?source=${serviceName}&reason=deactivated&retpath=${encodeURIComponent(originalUrl)}`);

        logger.info(`Forbidden ${originalUrl} because service "${serviceName}" deactivated`, req, req.context);
        res.redirect(redirectUrl);

        return;
    }

    next();
};
