const _ = require('lodash');
const sendHandledRequest = require('../../lib/sendHandledRequest');
const stackResponse = require('../../lib/stackResponse');

const GROUP_FIELDS = [
    'members_count', 'description', 'created', 'author', 'label',
    'email', 'members', 'type', 'name',
];

function create(req, res, next) {
    return sendHandledRequest({
        method: 'post',
        url: [req.config.api.directory, 'groups'],
        source: req,
        label: 'create_group',
    })
        .then(group => {
            stackResponse(req.context, 'groups', group);

            return next ? next() : req;
        })
        .catch(next);
}

function read(req, res, next) {
    return sendHandledRequest({
        method: 'get',
        url: [req.config.api.directory, 'groups', { value: req.query.id, type: 'number' }],
        source: req,
        label: 'read_group',
    })
        .then(group => {
            stackResponse(req.context, 'groups', group);

            return next ? next() : req;
        })
        .catch(next);
}

function readAdminGroup(req, res, next) {
    return sendHandledRequest({
        method: 'get',
        url: [req.config.api.directory, 'groups', '2'],
        source: req,
        label: 'read_group',
    })
        .then(group => {
            req.context.group = group;

            return next ? next() : req;
        })
        .catch(next);
}

function update(req, res, next) {
    return sendHandledRequest({
        method: 'patch',
        url: [req.config.api.directory, 'groups', { value: req.query.id, type: 'number' }],
        source: req,
        label: 'update_group',
    })
        .then(group => {
            stackResponse(req.context, 'groups', group);

            return next ? next() : req;
        })
        .catch(next);
}

function remove(req, res, next) {
    return sendHandledRequest({
        method: 'delete',
        url: [req.config.api.directory, 'groups', { value: req.query.id, type: 'number' }],
        source: req,
        label: 'remove_group',
    })
        .then(() => next ? next() : req)
        .catch(next);
}

// query: { id, type, admin_uid, page, per_page }
function readList(req, res, next) {
    const defaultOptions = {
        fields: GROUP_FIELDS.join(','),
        per_page: _.get(req.config, 'directory.perPage.groups'),
        ordering: 'name',
    };
    const query = _.extend(defaultOptions, req.query);

    return sendHandledRequest({
        method: 'get',
        url: [req.config.api.directory, 'v7', 'groups'],
        query,
        source: req,
        label: 'read_group_list',
    })
        .then(groupList => {
            const list = _.extend(groupList, { type: 'groups', query });

            stackResponse(req.context, 'lists', list);

            return next ? next() : req;
        })
        .catch(next);
}

function readMembers(req, res, next) {
    return sendHandledRequest({
        method: 'get',
        url: [req.config.api.directory, 'groups', { value: req.query.id, type: 'number' }, 'members'],
        source: req,
        label: 'read_group_members',
    })
        .then(groupMembers => {
            stackResponse(req.context, 'group_members', groupMembers);

            return next ? next() : req;
        })
        .catch(next);
}

module.exports = {
    create,
    read,
    update,
    remove,
    readList,
    readMembers,
    readAdminGroup,
};
