const express = require('express');

const nestedRouter = {
    org: require('./org')(),
};

module.exports = () => {
    const router = express.Router(); // eslint-disable-line

    const handleApiErrors = require('../../middlewares/handleApiErrors');
    const setRequestTime = require('../../middlewares/setRequestTime');
    const send = require('../../middlewares/send');

    const user = require('./extendedUser');
    const department = require('./extendedDepartment');
    const group = require('./extendedGroup');
    const organization = require('./extendedOrganization');
    const sender = require('./sender');

    const permissions = require('./permissions');
    const settings = require('./settings');
    const suggest = require('./suggest');
    const domains = require('./domains');
    const services = require('./services');
    const subscription = require('./subscription');
    const mail = require('./mail');
    const licenses = require('./licenses');
    const resources = require('./resources');
    const licenseRequests = require('./licenseRequests');
    const invite = require('./invite');
    const captcha = require('../../middlewares/captcha');
    const bunker = require('../../middlewares/bunker');

    router.route('/user')
        .post(user.create)
        .get(user.read)
        .put(user.update)
        .patch(user.update);

    router.route('/users')
        .get(user.readList)
        .post(user.updateList);

    router.route('/user/avatar')
        .post(user.setAvatar);

    router.route('/user/alias')
        .post(user.addAlias)
        .delete(user.removeAlias);

    router.route('/department')
        .post(department.create)
        .get(department.read)
        .put(department.update)
        .patch(department.update)
        .delete(department.remove);

    router.route('/departments')
        .get(department.readList);

    router.route('/department/members')
        .post(department.addMembers);

    router.route('/department/invite')
        .get(department.readLinkCode)
        .post(department.createLinkCode)
        .delete(department.removeLinkCode);

    router.route('/flatdepartment')
        .get(department.readFlat);

    router.route('/group')
        .post(group.create)
        .get(group.read)
        .put(group.update)
        .patch(group.update)
        .delete(group.remove);

    router.route('/groups')
        .get(group.readList);

    router.route('/group/members')
        .get(group.readMembers);

    router.route('/flatgroup')
        .get(group.readFlat);

    router.route('/organization')
        .get(organization.read)
        .put(organization.update)
        .patch(organization.update)
        .post(organization.create)
        .delete(organization.remove);

    router.route('/organization/domain')
        .post(organization.addDomain);

    router.route('/organization/new')
        .post(organization.createOrganizationWithoutDomain);

    router.route('/organization/bind')
        .post(organization.bindOrganizationWithResource);

    router.route('/organization/bind/organizations')
        .post(organization.readOrganizationsToBind);

    router.route('/organization/default-uid')
        .patch(organization.updateDefaultUid);

    router.route('/captcha')
        .get(captcha.generate);

    router.route('/bunker')
        .get(bunker);

    router.route('/invites/emails')
        .post(captcha.check, invite.inviteByEmails);

    router.route('/permissions')
        .get(permissions.read);

    router.route('/permissions/service')
        .get(permissions.readService);

    router.route('/settings')
        .get(settings.read)
        .put(settings.update)
        .delete(settings.remove);

    router.route('/settings/available')
        .get(settings.readKeys);

    router.route('/suggest')
        .get(suggest.read);

    router.route('/domains')
        .get(domains.read)
        .post(domains.add)
        .delete(domains.remove);

    router.route('/invites/use')
        .post(invite.createUser);

    router.route('/domains/ownership')
        .get(domains.getOwnershipInfo)
        .post(domains.checkOwnership);

    router.route('/sender/subscribe')
        .put(sender.subscribe);
    router.route('/sender/subscribe/request')
        .post(sender.request);

    router.route('/services')
        .get(services.readAll)
        .patch(services.update);

    router.route('/services/menu')
        .get(services.readMenu);

    router.route('/subscription')
        .get(subscription.readState);

    router.route('/subscription/pricing')
        .get(subscription.readPricing);

    router.route('/subscription/persons')
        .get(subscription.getPersons);

    router.route('/subscription/pay')
        .post(subscription.pay);

    router.route('/subscription/contract')
        .post(subscription.createContract);

    router.route('/subscription/promocodes/activate')
        .post(subscription.activatePromoCode);

    router.route('/subscription/licenses')
        .get(licenses.read)
        .put(licenses.update);

    router.route('/subscription/licenses/request')
        .get(licenseRequests.read)
        .post(licenseRequests.add);

    router.route('/subscription/licenses/request/deny/')
        .post(licenseRequests.deny);

    router.route('/subscription/licenses/request/confirm/')
        .post(licenseRequests.confirm);

    router.route('/subscription/licenses/preview')
        .get(licenses.getPreview);

    router.route('/subscription/calculate')
        .post(licenses.calculate);

    router.route('/resources')
        .get(resources.read)
        .delete(resources.remove);

    router.route('/resources/count')
        .get(resources.count);

    router.route('/resources/roles')
        .get(resources.roles);

    router.route('/resources/requests')
        .get(resources.requests);

    router.route('/resources/relations')
        .get(resources.readRelations);

    router.route('/resources/requests/confirm')
        .post(resources.confirmRequest);

    router.route('/resources/requests/deny')
        .post(resources.denyRequest);

    router.route('/relations')
        .post(resources.createRelation)
        .delete(resources.removeRelation);

    router.route('/mail/settings')
        .get(mail.getSettings)
        .post(mail.updateSettings);

    router.route('/mail/dkim/status')
        .get(mail.getDkimStatus);
    router.route('/mail/dkim/enable')
        .post(mail.enableDkim);

    router.route('/mail/check-import-server')
        .get(mail.checkImportServer);

    router.route('/mail/import')
        .post(mail.startImport)
        .get(mail.getImportStatus);

    router.use('/org', nestedRouter.org);

    router.use(handleApiErrors);
    router.use(setRequestTime);
    router.use(send);

    return router;
};
