const fs = require('fs');
const pathUtil = require('path');
const _ = require('lodash');

// source: https://stat.yandex-team.ru/DictionaryEditor/BlockStat
const blockStatDictionary = require('./blockstat_dictionary.json');
const keyMap = require('./key_map.json');

const DEFAULT_LAYOUT_ID = 'index';
const DEFAULT_GEO_ID = '213'; // Москва по геобазе

const RumEnvs = ['development', 'testing', 'prestable', 'production'];

function toBlockStatCode(id) {
    return (Array.isArray(id) ? id : id.split('.'))
        .map(component => blockStatDictionary[component])
        .filter(Boolean)
        .join('.');
}

function getInitData(req, layoutId) {
    const { context, config } = req;

    if (!layoutId) {
        layoutId = context.layoutId || DEFAULT_LAYOUT_ID;
    }

    layoutId = layoutId

        // sample_layout, sample-layout > sample.layout
        .replace(/[-_]/g, '.')

        // camelCaseHTMLLayout > camel.Case.HTMLLayout
        .replace(/([a-z])([A-Z])/g, '$1.$2')

        // camel.Case.HTMLLayout > camel.Case.HTML.Layout
        .replace(/([A-Z]+)([A-Z])([a-z])/g, '$1.$2$3')

        // camel.case.html.layout
        .toLowerCase();

    const pageId = `${config.rum.service_id}.layout.${layoutId}`;

    return {
        _id: pageId,
        settings: JSON.stringify({
            beacon: true,
            clck: config.rum.counter_url,
            slots: [], // в отсутсвие a/b-экспериментов - пустой массив
            reqid: _.get(context, 'headers.x-request-id'),
            crossOrigin: '',
        }),
        vars: JSON.stringify({
            // фиксируем идентификатор региона, пока нет необходимости
            // в статистике по регионам и подключении геобазы
            [keyMap.GEO_ID]: DEFAULT_GEO_ID,
            [keyMap.APP_PAGE_ID]: toBlockStatCode(pageId),
            '-project': config.rum.project_id,
            '-page': `layout.${layoutId}`,
            '-platform': 'desktop',
            '-env': RumEnvs.includes(config.app.env) ? config.app.env : 'development',
            '-version': config.app.version,
        }),
    };
}

// https://github.yandex-team.ru/RUM/rum-counter#Инлайновый-код
const INLINE_SCRIPTS = [
    '@yandex-int/rum-counter/dist/inline/interface.min.js',
    '@yandex-int/rum-counter/dist/inline/longtask.min.js',
    '@yandex-int/rum-counter/dist/inline/io.min.js',
];

module.exports = req => {
    const inlineScripts = INLINE_SCRIPTS
        .map(src => ({
            src,
            content: fs.readFileSync(pathUtil.join(__dirname, `../node_modules/${src}`))
                .toString()
                .trim(),
        }));

    return {
        service_id: req.config.rum.service_id,
        dictionary: blockStatDictionary,
        debug: Boolean(req.query.debug),
        init: getInitData(req),
        inline_scripts: inlineScripts,
    };
};
