/**
 * Скрипты сборки docker-образа
 */
const { basename, resolve } = require('path');
const { argv } = require('yargs');
const { cmd } = require('../../utils/cli-util');

const location = {
    codebaseRepo: 'https://github.yandex-team.ru/yandex-directory/connect.git',
    uploadRepo: 'registry.yandex.net/workspace/portal',
};

const buildDir = '.build';
const projectDir = basename(resolve(__dirname, '..'));

const { name: packageName } = require('../server/package.json');
let { version } = require('../server/package.json');

if (argv.v || argv.version) {
    version = argv.v || argv.version;
}

function log(message) {
    console.log(message); // eslint-disable-line no-console
}

function showHelpMsg() {
    log('Способы задания параметров сборки:');
    log('-v|--version <version> сборка с указаной версией');
    log('-h|--help справка по ключам');
    log('--feature сборка пакета для стенда');
    log('--release сборка пакета для релиза');
    log('--local локальная сборка');
}

if (argv.h || argv.help) {
    return showHelpMsg();
}

function getGitRepository() {
    if (argv.repository) {
        return argv.repository;
    }

    return location.codebaseRepo;
}

async function uploadStatic() {
    if (!argv.feature) {
        await cmd('YENV=production npm run ci:deploy:static', { verbose: true });
    }
}

async function buildDocker() {
    log('Собираем docker-образ');

    await cmd(`docker build -t ${location.uploadRepo}:${version} .`, { verbose: true });

    log('Сборка docker-образа завершена');
}

async function uploadDocker() {
    log(`Загружаем docker-образ в репозиторий '${location.uploadRepo}'`);

    await cmd(`docker push ${location.uploadRepo}:${version}`, { verbose: true });

    log('Загрузка docker-образа завершена');
    log('Сборка доступна в Qloud (https://qloud-ext.yandex-team.ru/projects/workspace)');
}
async function buildStatic() {
    await cmd('npm ci && npm run prod', { verbose: true });
}

async function buildPackage() {
    await buildStatic();
    await uploadStatic();
    await buildDocker();
    await uploadDocker();
}

async function checkoutVersion() {
    let branch = version;

    if (!argv.feature) {
        branch = `${packageName}-${version}`;
    }

    await cmd([
        `rm -rf ${buildDir}`,
        `git clone --branch ${branch} ${getGitRepository()} ${buildDir}`,
    ]);
}

async function cleanup() {
    await cmd(`rm -rf ${buildDir}`);
}

(async() => {
    if (argv.h || argv.help) {
        return showHelpMsg();
    }

    if (!version) {
        showHelpMsg();

        throw new Error('Не задана версия сборки');
    }

    if (!argv.feature && !argv.release) {
        showHelpMsg();

        throw new Error('Не задан тип сборки');
    }

    if (argv.local) {
        const buildParams = process.argv
            .slice(2)
            .filter(param => param !== '--local')
            .join(' ');

        await checkoutVersion();
        await cmd(`cd ${buildDir}/${projectDir} && ./deps.sh`, { verbose: true });
        await cmd(`cd ${buildDir}/${projectDir} && npm run build -- ${buildParams}`, { verbose: true });
        await cleanup();
    } else {
        await buildPackage();
    }

    log('✔ Выполнено');
    log(`Версия сборки: ${version}`);
})();
