"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = _default;
exports.messages = exports.ruleName = void 0;

var _utils = require("../../utils");

var _stylelint = require("stylelint");

var _lodash = require("lodash");

var ruleName = (0, _utils.namespace)("dollar-variable-empty-line-before");
exports.ruleName = ruleName;

var messages = _stylelint.utils.ruleMessages(ruleName, {
  expected: "Expected an empty line before $-variable",
  rejected: "Unexpected empty line before $-variable"
});

exports.messages = messages;

function _default(expectation, options, context) {
  return function (root, result) {
    var validOptions = _stylelint.utils.validateOptions(result, ruleName, {
      actual: expectation,
      possible: ["always", "never"]
    }, {
      actual: options,
      possible: {
        except: ["first-nested", "after-comment", "after-dollar-variable"],
        ignore: ["after-comment", "inside-single-line-block"],
        disableFix: _lodash.isBoolean
      },
      optional: true
    });

    if (!validOptions) {
      return;
    }

    var fix = function fix(decl, match, replace) {
      decl.raws.before = decl.raws.before.replace(new RegExp("^".concat(match)), replace);
    };

    var hasNewline = function hasNewline(str) {
      return str.indexOf(context.newline) > -1;
    };

    root.walkDecls(function (decl) {
      if (!isDollarVar(decl)) {
        return;
      } // Always ignore the first $var in a stylesheet


      if (decl === root.first) {
        return;
      } // If ignoring vars after comments is set


      if ((0, _utils.optionsHaveIgnored)(options, "after-comment") && decl.prev() && decl.prev().type === "comment") {
        return;
      } // If ignoring single-line blocks


      if ((0, _utils.optionsHaveIgnored)(options, "inside-single-line-block") && decl.parent.type !== "root" && (0, _utils.isSingleLineString)((0, _utils.blockString)(decl.parent))) {
        return;
      }

      var expectHasEmptyLineBefore = expectation === "always"; // Reverse for a variable that is a first child of its parent

      if ((0, _utils.optionsHaveException)(options, "first-nested") && decl === decl.parent.first) {
        expectHasEmptyLineBefore = !expectHasEmptyLineBefore;
      } // Reverse if after a comment


      if ((0, _utils.optionsHaveException)(options, "after-comment") && decl.prev() && decl.prev().type === "comment") {
        expectHasEmptyLineBefore = !expectHasEmptyLineBefore;
      } // Reverse if after another $-variable


      if ((0, _utils.optionsHaveException)(options, "after-dollar-variable") && decl.prev() && isDollarVar(decl.prev())) {
        expectHasEmptyLineBefore = !expectHasEmptyLineBefore;
      }

      var before = decl.raws.before;

      if (expectHasEmptyLineBefore === (0, _utils.hasEmptyLine)(before)) {
        return;
      }

      var isFixDisabled = options && options.disableFix === true;

      if (context.fix && !isFixDisabled) {
        if (expectHasEmptyLineBefore && !(0, _utils.hasEmptyLine)(before)) {
          fix(decl, context.newline, context.newline + context.newline);

          if ((0, _utils.optionsHaveException)(options, "first-nested") && !hasNewline(before)) {
            fix(decl, "\\s+", context.newline + context.newline);
          }

          return;
        }

        if (!expectHasEmptyLineBefore && (0, _utils.hasEmptyLine)(before)) {
          fix(decl, "\\n\\r\\n", "\r\n");
          fix(decl, context.newline + context.newline, context.newline);
          return;
        }
      }

      _stylelint.utils.report({
        message: expectHasEmptyLineBefore ? messages.expected : messages.rejected,
        node: decl,
        result: result,
        ruleName: ruleName
      });
    });
  };
}

function isDollarVar(node) {
  return node.prop && node.prop[0] === "$";
}