// vanilla js

(function() {
    // Ничего не делаем, если подключаем в серверном коде
    if (typeof window === 'undefined') {
        return;
    }

    window.Ya = window.Ya || {};

    let xivaBaseParams = {
        api: 'wss://push.yandex.ru/v2/subscribe/websocket',
        reconnectMaxTimeout: 2000,
        reconnectCooldown: 10000,
        reconnectTimeoutIncrement: 500,
    };

    /**
     * @description Создаёт подключение к WebSocket почтовой Ксивы Яндекса
     *
     * @param {Object} params параметры
     * @param {String} [params.api] урл к ручке, не должен содержать GET параметры
     * @param {String} [params.reconnectMaxTimeout] максимальное время задержки до переподключения
     * (рандомно берёт значение этом диапазоне)
     * @param {String} [params.reconnectCooldown] время сброса задержки для переподключения
     * @param {String} [params.reconnectTimeoutIncrement] значение которое будет добавлено к задержке
     * перед следующим подключением
     * @param {String} [params.urlParams] GET параметры к ручке
     *
     * @returns {Object} экземпляр YA XIVA
     */
    let XIVA = window.Ya.XIVA = function(params) {
        this._params = params || {};
        this._params.urlParams = this._params.urlParams || {};

        Object.keys(xivaBaseParams).forEach(function(v) {
            this._params[v] = this._params[v] || xivaBaseParams[v];
        }, this);

        this._eventer = document.createDocumentFragment();
        this.addEventListener = this._eventer.addEventListener.bind(this._eventer);
        this.removeEventListener = this._eventer.removeEventListener.bind(this._eventer);

        this._setReconnectTimeout();
        this._onOpen = this._onOpen.bind(this);
        this._onClose = this._onClose.bind(this);
        this._onMessage = this._onMessage.bind(this);
        this._onError = this._onError.bind(this);

        this._connect();
    };

    /**
     * @description Переподключение с другими параметрами
     *
     * @param {Object} urlParams новые GET параметры к ручке
     * @param {Boolean} fullRewrite перезаписать все старые параметры
     */
    XIVA.prototype.reconnect = function(urlParams, fullRewrite) {
        if (!urlParams) {
            return;
        }

        if (fullRewrite) {
            this._params.urlParams = urlParams;
        } else {
            Object.keys(urlParams).forEach(function(key) {
                this._params.urlParams[key] = urlParams[key];
            }, this);
        }

        this._connect();
    };

    /**
     * @description Закрывает канал и сбрасывает события
     */
    XIVA.prototype.destruct = function() {
        let ws = this._ws;

        this._destructing = true;

        this._unbindeEvents();
        ws.close();
    };

    /**
     * @description Отправляет сообщение в ручку
     *
     * @param {String} message
     */
    XIVA.prototype.send = function(message) {
        this._ws.send(message);
    };

    // PRIVATE

    /**
     * @description Подключается к ручке с параметрами из this._params
     *
     * @private
     */
    XIVA.prototype._connect = function() {
        if (typeof WebSocket !== 'function') {
            return;
        }

        this._unbindeEvents();

        if (this._ws) {
            this._ws.close();
        }

        this._ws = new WebSocket(this._prepareUrl());

        this._bindEvents();
    };

    /**
     * @description Генерирует правильный url с параметрами из this._params
     *
     * @private
     */
    XIVA.prototype._prepareUrl = function() {
        let url = this._params.api;
        let params = this._params.urlParams;
        let result = [];
        let prefix = '?';

        if (url.indexOf('?') !== -1) {
            prefix = '&';
        }

        Object.keys(params).forEach(function(key) {
            if (params[key]) {
                result.push(key + '=' + encodeURIComponent(params[key]));
            }
        });

        if (result.length) {
            return url + prefix + result.join('&');
        }
        return url;
    };

    /**
     * @description Сбрасывает слушателей событий
     *
     * @private
     */
    XIVA.prototype._unbindeEvents = function() {
        let ws = this._ws;

        if (ws) {
            ws.removeEventListener('open', this._onOpen);
            ws.removeEventListener('close', this._onClose);
            ws.removeEventListener('message', this._onMessage);
            ws.removeEventListener('error', this._onError);
        }
    };

    /**
     * @description Добавляет слушателей событий
     *
     * @private
     */
    XIVA.prototype._bindEvents = function() {
        let ws = this._ws;

        ws.addEventListener('open', this._onOpen);
        ws.addEventListener('close', this._onClose);
        ws.addEventListener('message', this._onMessage);
        ws.addEventListener('error', this._onError);
    };

    /**
     * @description Обработчик собыитя open
     *
     * @private
     */
    XIVA.prototype._onOpen = function(data) {
        clearTimeout(this._reconnectCooldown);

        this._reconnectCooldown = setTimeout(function() {
            this._setReconnectTimeout();
        }.bind(this), this._params.reconnectCooldown);

        this._dispatch('open', data);
    };

    /**
     * @description Обработчик собыитя close
     *
     * @private
     */
    XIVA.prototype._onClose = function(data) {
        if (!this._destructing) {
            setTimeout(this._connect.bind(this), this._reconnectTimeout);

            this._reconnectTimeout += this._params.reconnectTimeoutIncrement;
        }

        this._dispatch('close', data);
    };

    /**
     * @description Обработчик собыитя message
     *
     * @private
     */
    XIVA.prototype._onMessage = function(message) {
        let data;

        try {
            data = JSON.parse(message.data);
        } catch (e) {
            return;
        }

        this._dispatch('message', data);
    };

    /**
     * @description Обработчик события error
     *
     * @private
     */
    XIVA.prototype._onError = function(data) {
        this._dispatch('error', data);
    };

    /**
     * @description Диспетчер событий
     *
     * @private
     */
    XIVA.prototype._dispatch = function(eventName, data) {
        let event = document.createEvent('CustomEvent');

        event.initCustomEvent(eventName, null, null, data);

        this._eventer.dispatchEvent(event);
    };

    /**
     * @description Задаёт базовое время переподключения, чтобы не завалить сервер
     * делаем это в немного случайный момент времени
     *
     * @private
     */
    XIVA.prototype._setReconnectTimeout = function() {
        this._reconnectTimeout = Math.round(Math.random() * this._params.reconnectMaxTimeout);
    };
})();
