/* global BEM, $ */

/**
 * @file
 * Блок, обеспечивающий сбор и отправку данных RUM (Real User Measurements) для AJAX-запросов.
 *
 * Подписывается в канале ajax на события
 * - ajaxAction - событие действия, приводящего к загрузке страницы аяксом
 * - successCallbacksStarted - начало выполнения обработчиков AJAX, (в основном) обновляющих DOM
 * - successCallbacksFinished - окончание выполнения обработчиков AJAX, (в основном) обновляющих DOM
 *
 * Подписывается на глобальные события jQuery AJAX: ajaxSend, ajaxSuccess, ajaxError, ajaxComplete.
 *
 * Является связью между Ya.Rum.AjaxPerf и блоками i-bem
 */
BEM.decl('rum__bem-ajax', {
    onSetMod: {
        js: function() {
            var that = this;

            this._requests = {};

            BEM.channel('ajax').on({
                ajaxAction: function() {
                    that._actionTime = Ya.Rum.getTime();

                    Ya.Rum.sendTrafficData();
                    Ya.Rum.finalizeLayoutShiftScore();
                    Ya.Rum.finalizeLargestContentfulPaint();
                },

                successCallbacksStarted: function() {
                    that._renderStart = Ya.Rum.getTime();
                },

                successCallbacksFinished: function() {
                    if (that._renderStart) {
                        that._renderDuration = Ya.Rum.getTime() - that._renderStart;
                        that._renderStart = null;
                    }
                }
            });

            BEM.DOM.doc.on({
                ajaxSend: this._onSend.bind(this),
                ajaxSuccess: this._onSuccess.bind(this),
                ajaxError: this._onError.bind(this),
                ajaxComplete: this._onComplete.bind(this)
            });
        }
    },

    /**
     * Возвращает экземпляр Ya.Rum.AjaxPerf, соответствующий указанному XHR
     *
     * @param {XMLHttpRequest} xhr
     *
     * @returns {Ya.Rum.AjaxPerf}
     */
    _getRequestTracker: function(xhr) {
        return this._requests[$.identify(xhr)];
    },

    /**
     * Обработчик начала AJAX-запроса
     *
     * @param {Event} e
     * @param {XMLHttpRequest} xhr
     * @param {Object} options
     */
    _onSend: function(e, xhr, options) {
        options = options || {};

        if (this._shouldTrackRequest(options)) {
            this._requests[$.identify(xhr)] = new Ya.Rum.AjaxPerf({
                xhr: xhr,
                url: options.url,
                actionTime: this._actionTime
            });
            this._actionTime = null;
        }
    },

    /**
     * Обработчик успеха AJAX-запроса
     *
     * @param {Event} e
     * @param {XMLHttpRequest} xhr
     * @param {Object} options
     * @param {Object} data
     */
    _onSuccess: function(e, xhr, options, data) {
        if (data && data.error) {
            var ajaxPerf = this._getRequestTracker(xhr);

            if (ajaxPerf) {
                ajaxPerf.onRequestError('Handled: ' + data.error);
            }
        }
    },

    /**
     * Обработчик ошибки AJAX-запроса
     *
     * @param {Event} e
     * @param {XMLHttpRequest} xhr
     * @param {Object} options
     * @param {String} error
     */
    _onError: function(e, xhr, options, error) {
        var ajaxPerf = this._getRequestTracker(xhr);

        if (ajaxPerf) {
            ajaxPerf.onRequestError(error);
        }
    },

    /**
     * Обработчик окончания AJAX-запроса
     *
     * @param {Event} e
     * @param {XMLHttpRequest} xhr
     */
    _onComplete: function(e, xhr) {
        var ajaxPerf = this._getRequestTracker(xhr);

        if (!ajaxPerf) {
            return;
        }

        ajaxPerf.onRequestComplete();
        if (this._renderDuration) {
            ajaxPerf.onRenderComplete(this._renderDuration);
        }

        delete this._requests[$.identify(xhr)];
        this._renderDuration = null;

        this.afterCurrentEvent(function() {
            ajaxPerf.send();
        });
    },

    /**
     * Нужно ли отслеживать этот запрос и определять его временные характеристики?
     *
     * @param {{url}} options
     *
     * @returns {Boolean}
     */
    _shouldTrackRequest: function(options) {
        var uri = BEM.blocks.uri.parse(options.url),
            params = uri.queryParams;

        return this._validateUrl(uri) &&
            params.ajax && !(params.rpt || params.filter);
    },

    /**
     * Нужно ли отправлять счётчик для этого url?
     * Заглушка, переопределяется на нижележащих уровнях (blocks-desktop, blocks-touch-phone, blocks-touch-pad)
     *
     * @param {Object} uri
     *
     * @returns {Boolean}
     */
    _validateUrl: function(uri) {
        return false;
    }
}, {
    /**
     * Создать единственный экземпляр
     *
     * @returns {BEM}
     */
    getInstance: function() {
        return this._instance || (this._instance = BEM.create('rum__bem-ajax'));
    }
});
