(function () {
'use strict';

var rum = Ya.Rum;
var PATH_AJAX_PERF = '690.1201'; // tech.ajax

/**
 * Счётчик скорости аякса
 * @constructor
 *
 * @param {Object} options Параметры объекта
 *
 * @param {Number} [options.actionTime] Время действия пользователя (клик или другое действие, приводящее к AJAX) –
 *    нужно для метрик ajax.before и ajax.total
 * @param {Number} [options.requestSendTime] Специфичное время отправки запроса;
 *    если отсутствует за него принимается время создания объекта
 * @param {String} [options.url] URL запроса, нужен для извлечения данных из Resource Timing API
 * @param {XMLHttpRequest} [options.xhr] XHR запроса, нужен для передачи статуса запроса в счётчике
 */
function AjaxPerf(options) {
    options = options || {};

    this._sendTime = options.requestSendTime || rum.getTime();
    rum.ajaxStart = this._ajaxStartTime = options.actionTime || rum.getTime();
    rum.ajaxComplete = 0;

    this._actionTime = options.actionTime;
    this._xhr = options.xhr;
    this._url = options.url;
}

AjaxPerf.prototype = /** @lends AjaxPerf.prototype */ {
    constructor: AjaxPerf,

    /**
     * Колбэк ошибки запроса
     *
     * @param {String|Error} errorType
     */
    onRequestError: function(errorType) {
        this._errorTime = rum.getTime();
        this._errorType = errorType;
    },

    /**
     * Колбэк окончания запроса
     */
    onRequestComplete: function() {
        this._completeTime = rum.getTime();
    },

    /**
     * Колбэк окончания рендеринга результата
     *
     * @param {Number} duration
     */
    onRenderComplete: function(duration) {
        this._renderDuration = duration;
    },

    /**
     * Отправить счётчик
     */
    send: function() {
        var that = this;

        rum.ajaxComplete = that._errorTime || that._completeTime;
        if (!rum.ajaxComplete) {
            return;
        }

        var xhr = that._xhr;
        var vars = [
            // xhr.status
            xhr && ('2772.720=' + xhr.status),

            // xhr.status.text
            xhr && ('2772.720.232=' + xhr.statusText),

            // ajax.total
            '1201.906=' + rum.normalize(rum.ajaxComplete, that._ajaxStartTime),

            // ajax.action
            that._actionTime && ('1201.689=' + rum.normalize(that._actionTime)),

            // ajax.start
            that._ajaxStartTime && ('1201.2154=' + rum.normalize(that._ajaxStartTime)),

            // ajax.complete
            rum.ajaxComplete && ('1201.3103=' + rum.normalize(rum.ajaxComplete)),

            // ajax.before
            that._actionTime && ('1201.789=' + rum.normalize(that._sendTime, that._actionTime)),

            // ajax.dom
            that._renderDuration && ('1201.1310=' + rum.normalize(that._renderDuration)),

            // [device_memory] Объём памяти устройства, см. https://github.com/w3c/device-memory
            navigator.deviceMemory && '3140=' + navigator.deviceMemory,

            // [hardware_concurrency] Количество ядер, см. https://clck.ru/DYqo9
            navigator.hardwareConcurrency && '3141=' + navigator.hardwareConcurrency
        ];

        rum.pushConnectionTypeTo(vars);

        function sendAjaxPerf() {
            var url = that._errorTime && that._url ? that._url : null;
            var commonVars = rum.getVarsList().concat(
                // [ua] User-agent
                '1042=' + encodeURIComponent(navigator.userAgent)
            );
            rum.sendCounter(PATH_AJAX_PERF, commonVars.concat(vars), null, url);
        }

        if (that._errorTime) {
            vars.push(
                // ajax.error
                '1201.1030=1',

                // ajax.error.message
                that._errorType && '1201.1030.1304=' + that._errorType
            );

            sendAjaxPerf();
        } else if (that._url) {
            rum.getResourceTimings(that._url, function(resTimings) {
                if (resTimings) {
                    var resTiming = resTimings[0];

                    vars.push(
                        '1036=' + (resTiming.domainLookupStart - resTiming.startTime), // wait
                        '1037=' + (resTiming.domainLookupEnd - resTiming.domainLookupStart), // dns
                        '1038=' + (resTiming.connectEnd - resTiming.connectStart), // tcp
                        '1039=' + (resTiming.responseStart - resTiming.connectEnd), // ttfb
                        '1040=' + (resTiming.responseEnd - resTiming.responseStart), // html
                        '1040.906=' + (resTiming.responseEnd - resTiming.domainLookupStart), // html.total
                        (resTiming.secureConnectionStart > 0) &&
                        ('1383=' + (resTiming.connectEnd - resTiming.secureConnectionStart)) // ssl
                    );

                    rum.pushTimingTo(vars, resTiming);
                }

                sendAjaxPerf();
            });
        } else {
            sendAjaxPerf();
        }
    }
};

rum.AjaxPerf = AjaxPerf;

}());
