(function() {
    var rum = Ya.Rum;

    if (!rum.enabled || !window.PerformanceObserver) {
        return;
    }

    var PATH_IMAGES = '690.2096.277'; // tech.perf.images

    var SIZE_THRESHOLD = 10;
    var SEARCH_LAST_TIMEOUT = 15000;
    var DATA_URI_ID = 'data-uri';

    var resourcesExpected = 0;
    var resourcesFound = 0;
    var hasDataUri = false;
    var hasUnknown = false;

    var docRespEnd;
    var firstTime;
    var lastTime;

    rum.onReady(main);

    function main() {
        var timing = performance.timing;
        docRespEnd = timing.responseEnd - timing.navigationStart;

        var targetElements = getElements();

        requestAnimationFrame(function() {
            if (!rum.getSetting('forcePaintTimeSending') && rum.isVisibilityChanged()) {
                return;
            }

            var visibleElements = filterVisibleElements(targetElements);
            try {
                handleVisibleElements(visibleElements);
            } catch (e) {}
        });
    }

    function getElements() {
        var candidateElements = document.querySelectorAll('img,[data-rcid],[style*="background"]');

        var targetElements = [];

        for (var i = 0; i < candidateElements.length; i++) {
            var elem = candidateElements[i];
            if (elem.loading === 'lazy') {
                continue;
            }

            var url;

            if (elem.src) {
                url = elem.src;
            } else {
                url = getComputedStyle(elem).backgroundImage;
                if (url.indexOf('url(') === 0) {
                    url = url.slice(4, -1);
                    if (url[0] === '"' || url[0] === "'") {
                        url = url.slice(1, -1);
                    }
                }
            }

            if (url) {
                if (url.indexOf('data:') === 0) {
                    url = DATA_URI_ID;
                }
                targetElements.push({ elem: elem, url: url });
            }
        }

        return targetElements;
    }

    function filterVisibleElements(elements) {
        var docElem = document.documentElement;

        var vWidth = docElem.clientWidth;
        var vHeight = docElem.clientHeight;

        var visibleElements = [];

        for (var i = 0; i < elements.length; i++) {
            var elemData = elements[i];
            var rect = elemData.elem.getBoundingClientRect();
            if (rect.width < SIZE_THRESHOLD || rect.height < SIZE_THRESHOLD) {
                continue;
            }

            if (rect.bottom <= 0 || rect.top >= vHeight || rect.right <= 0 || rect.left >= vWidth) {
                continue;
            }

            visibleElements.push(elemData);
        }

        return visibleElements;
    }

    function handleVisibleElements(elements) {
        if (!elements.length) {
            return;
        }

        var normalizeLink = document.createElement('link');

        var urlSet = {};

        for (var i = 0; i < elements.length; i++) {
            var elemData = elements[i];

            if (elemData.url === DATA_URI_ID) {
                hasDataUri = true;
                continue;
            }

            normalizeLink.href = elemData.url;
            urlSet[normalizeLink.href] = true;
        }

        if (hasDataUri) {
            // Условность – считаем, что data URI загружаются тогда же, когда документ
            firstTime = docRespEnd;
            lastTime = docRespEnd;
        }

        resourcesExpected = Object.keys(urlSet).length;

        if (resourcesExpected === 0) {
            return finalizeStats('e0');
        }

        var finalizeTimeout;

        var po = new PerformanceObserver(function(perfEntryList) {
            var entries = perfEntryList.getEntries();

            for (var i = 0; i < entries.length; i++) {
                var entry = entries[i];
                if (!urlSet[entry.name]) {
                    continue;
                }

                var time = entry.responseEnd;
                if (!time) {
                    // Случай, когда данные недоступны
                    // Условно считаем, что загрузилось через duration после загрузки документа
                    hasUnknown = true;
                    time = docRespEnd + entry.duration;
                }

                if (firstTime === undefined || time < firstTime) {
                    firstTime = time;
                }

                if (lastTime === undefined || time > lastTime) {
                    lastTime = time;
                }

                if (++resourcesFound === resourcesExpected) {
                    po.disconnect();
                    finalizeStats('ok');
                    clearTimeout(finalizeTimeout);
                }
            }
        });

        po.observe({
            type: 'resource',
            buffered: true
        });

        finalizeTimeout = setTimeout(function() {
            try {
                po.disconnect();
                finalizeStats('tm');
            } catch (e) {}
        }, SEARCH_LAST_TIMEOUT);
    }

    function finalizeStats(finalizeType) {
        rum.sendCounter(PATH_IMAGES, [
            'ft=' + rum.normalize(firstTime),
            'lt=' + rum.normalize(lastTime),
            'er=' + resourcesExpected,
            'fr=' + resourcesFound,
            't=' + (finalizeType || 'unk'),
            hasDataUri && 'd=1',
            hasUnknown && 'u=1'
        ]);
    }
})();
