/**
 * @file
 * Реализация методов счётчиков скорости
 * Необходимо предварительно включить в код страницы реализацию интерфейса Ya.Rum
 *
 * Код написан так, чтобы обеспечить наилучшее сжатие uglify + gzip. Проверены:
 * - IIFE (function(window, document, navigator, undefined) { - no profit
 * - var getKeys = Object.keys - no profit
 */
(function(rum, undefined) {
    if (!rum) {
        throw new Error('Rum: interface is not included');
    }

    if (!rum.enabled) {
        rum.getSetting = function() {
            return '';
        };

        rum.getVarsList = function() {
            return [];
        };

        rum.getResourceTimings = rum.pushConnectionTypeTo = rum.pushTimingTo = rum.normalize = rum.sendCounter =
            rum.sendDelta = rum.sendTimeMark = rum.sendResTiming = rum.sendTTI = rum.makeSubPage =
                rum.sendHeroElement = rum.onReady = function() {};

        return;
    }

    /**
     * Получить переданные при инициализации RUM параметры как список строк вида 'key=value'.
     *
     * @returns {Array<String>}
     */
    rum.getVarsList = function() {
        var vars = rum._vars;
        return Object.keys(vars).map(function(varName) {
            return varName + '=' +  encodeURIComponent(vars[varName]).replace(/\*/g, '%2A');
        });
    };

    /**
     * Переопределяет RUM параметры
     *
     * @param {Object} vars – параметры, которые нужно переопределить
     */
    rum.setVars = function(vars) {
        Object.keys(vars).forEach(function (k) {
            rum._vars[k] = vars[k];
        });

        updateCommonVars();
        updateMarkCommonVars();
    };

    var PATH_TECH_TIMING = '690.1033'; // tech.timing
    var PATH_TIME = '690.2096.207'; // tech.perf.time
    var PATH_DELTA = '690.2096.2877'; // tech.perf.delta
    var PATH_NAVIGATION = '690.2096.2892'; // tech.perf.navigation
    var PATH_RESOURCE_TIMING = '690.2096.2044'; // tech.perf.resource_timing
    var TECH_PERF_TRAFFIC = '690.2096.361'; // tech.perf.traffic
    var TECH_PERF_CLS = '690.2096.4004'; // tech.perf.cls

    var POLLING_ITERATIONS_LIMIT = 10;

    var LONG_TASK_TIMEOUT = 3000;
    var LONG_TASK_LIMIT = 20000;

    var COMMON_TIMING_PROPS_MAP = {
        connectEnd: 2116,
        connectStart: 2114,
        decodedBodySize: 2886,
        domComplete: 2124,
        domContentLoadedEventEnd: 2131,
        domContentLoadedEventStart: 2123,
        domInteractive: 2770,
        domLoading: 2769,
        domainLookupEnd: 2113,
        domainLookupStart: 2112,
        duration: 2136,
        encodedBodySize: 2887,
        entryType: 2888,
        fetchStart: 2111,
        initiatorType: 2889,
        loadEventEnd: 2126,
        loadEventStart: 2125,
        nextHopProtocol: 2890,
        redirectCount: 1385,
        redirectEnd: 2110,
        redirectStart: 2109,
        requestStart: 2117,
        responseEnd: 2120,
        responseStart: 2119,
        secureConnectionStart: 2115,
        startTime: 2322,
        transferSize: 2323,
        type: 76,
        unloadEventEnd: 2128,
        unloadEventStart: 2127,
        workerStart: 2137
    };

    var VISIBILITY_STATES = {
        visible: 1,
        hidden: 2,
        prerender: 3
    };

    var CONNECTION_TYPES = {
        bluetooth: 2064,
        cellular: 2065,
        ethernet: 2066,
        none: 1229,
        wifi: 2067,
        wimax: 2068,
        other: 861,
        unknown: 836,

        // Legacy
        0: 836, // unknown
        1: 2066, // ethernet
        2: 2067, // wifi
        3: 2070, // 2G
        4: 2071, // 3G
        5: 2768 // 4G
    };

    var PAINT_TYPES = {
        'first-paint': 2793,
        'first-contentful-paint': 2794
    };
    var PAINT_EVENTS_COUNT = Object.keys(PAINT_TYPES).length;

    var getTime = rum.getTime;

    var PerfObserver = window.PerformanceObserver;

    var perf = window.performance || {};
    var pageTiming = perf.timing || {};
    var navigation = perf.navigation || {};
    var connection = navigator.connection;

    // хранилище отправленных временных меток
    var timeMarks = {};

    var trafficDataBuffer = {};
    var deltaMarks = rum._deltaMarks;

    var normalizeLink = document.createElement('link');
    var normalizeAnchor = document.createElement('a');

    var resourceTimingSupported = typeof perf.getEntriesByType === 'function';

    var navStart = pageTiming.navigationStart;

    var commonVars,
        markCommonVars;

    updateCommonVars();
    updateMarkCommonVars();

    // Времена последнего AJAX-запроса (если был)
    rum.ajaxStart = 0;
    rum.ajaxComplete = 0;

    function updateCommonVars() {
        commonVars = rum.getVarsList();
        if (rum.getSetting('sendClientUa')) {
            // [ua] User-agent
            commonVars.push('1042=' + encodeURIComponent(navigator.userAgent));
        }
    }

    function updateMarkCommonVars() {
        markCommonVars = commonVars.concat([
            '143.2129=' + navStart // page.navigation_start
        ]);
    }

    /**
     * Отправить данные Navigation Timing Level 2.
     * @see https://www.w3.org/TR/navigation-timing-2/
     * @see https://developer.mozilla.org/en-US/docs/Web/API/PerformanceObserver
     */
    if (PerfObserver) {
        // Согласно спецификации https://clck.ru/CEM5D
        // если браузер поддерживает PerformanceObserver, но в entryTypes
        // используется только тип который не распознается – будет брошено
        // исключение "entryTypes не может быть пустым"
        try {
            (new PerfObserver(function sendNavigationTiming(list, observer) {
                var navigationTiming = list.getEntriesByType('navigation')[0];
                if (!navigationTiming) {
                    return;
                }

                // https://st.yandex-team.ru/SERP-91162
                if (observer) {
                    observer.disconnect();
                }

                var vars = [];
                pushTimingTo(vars, navigationTiming);
                pushConnectionTypeTo(vars);

                sendCounter(PATH_NAVIGATION, commonVars.concat(vars));
            })).observe({entryTypes: ['navigation']});
        } catch (e) {
        }
    }

    // Init
    onReady(main);

    /**
     * Возвращает базовые параметры
     * @returns {Array<String>}
     * @protected
     */
    rum._getCommonVars = function() {
        return commonVars;
    };

    /**
     * Вызвать фунцию, когда страница будет готова
     * @param {Function} cb
     */
    function onReady(cb) {
        function wrappedCb() {
            removeEventListener('DOMContentLoaded', wrappedCb);
            removeEventListener('load', wrappedCb);
            cb();
        }

        if (document.readyState === 'loading') {
            addEventListener('DOMContentLoaded', wrappedCb);
            addEventListener('load', wrappedCb);
        } else {
            cb();
        }
    }

    function main() {
        if (!navStart) {
            return;
        }

        setTimeout(function() {
            // Заменяем методы, записывающие данные в массив, на методы, сразу отправляющие данные на бэкэнд
            rum.sendTimeMark = sendTimeMark;
            rum.sendResTiming = sendResTiming;
            rum.timeEnd = timeEnd;

            var defRes = rum._defRes;
            while (defRes.length) {
                var resData = defRes.shift();
                sendResTiming(resData[0], resData[1]);
            }

            var defTimes = rum._defTimes;
            while (defTimes.length) {
                var timeData = defTimes.shift();
                sendTimeMark(timeData[0], timeData[1], false, timeData[2]);
            }

            Object.keys(deltaMarks).forEach(function(counterId) {
                sendDelta(counterId);
            });

            sendPageLoadingTiming();
            sendPaintTiming();
            if (rum.getSetting('sendAutoElementTiming')) {
                sendElementTiming();
            }

            checkLongTasks();

            if (document.readyState === 'complete') {
                runOnLoadTasks();
            } else {
                addEventListener('load', runOnLoadTasks);
            }
        }, 0);
    }

    function runOnLoadTasks() {
        if (rum.getSetting('disableOnLoadTasks')) {
            return;
        }

        removeEventListener('load', runOnLoadTasks);

        initPeriodicStats();
        initTrafficStats();
        sendFirstInputDelay();
        calcLayoutShiftScore();
        calcLargestContentfulPaint();
    }

    //
    // Методы для измерения загрузки страницы
    //

    /**
     * Отправить тайминги загрузки документа.
     *
     * В Firefox в состоянии readyState === 'interactive' остаются нулевыми времена
     * domContentLoadedEventStart = domContentLoadedEventEnd === 0, поэтому ждём ненулевых значений.
     * в Chrome и Safari domContentLoadedEventStart и domContentLoadedEventEnd дают ожидаемые значения без ожидания.
     */
    function sendPageLoadingTiming() {
        var domContentLoadedEventStart = pageTiming.domContentLoadedEventStart,
            domContentLoadedEventEnd = pageTiming.domContentLoadedEventEnd;

        if (domContentLoadedEventStart === 0 && domContentLoadedEventEnd === 0) {
            setTimeout(sendPageLoadingTiming, 50);
            return;
        }

        var vars = commonVars.concat([
            // [navigation_start] Время старта запроса
            '2129=' + navStart,

            // [wait] Время ожидания перед запросом
            '1036=' + (pageTiming.domainLookupStart - navStart),

            // [dns] Время DNS Lookup
            '1037=' + (pageTiming.domainLookupEnd - pageTiming.domainLookupStart),

            // [tcp] Время TCP Handshake
            '1038=' + (pageTiming.connectEnd - pageTiming.connectStart),

            // [ssl] Время установки безопасного соединения
            pageTiming.secureConnectionStart && '1383=' + (pageTiming.connectEnd - pageTiming.secureConnectionStart),

            // [ttfb] Время до первого байта
            '1039=' + (pageTiming.responseStart - pageTiming.connectEnd),

            // [html] Время получения HTML от первого байта
            '1040=' + (pageTiming.responseEnd - pageTiming.responseStart),

            // [html.total] Время получения HTML от DNS Lookup
            '1040.906=' + (pageTiming.responseEnd - pageTiming.domainLookupStart),

            // [dom.loading] Время начала парсинга HTML от первого байта
            '1310.2084=' + (pageTiming.domLoading - pageTiming.responseStart),

            // [dom.interactive] Время окончания парсинга HTML от первого байта
            '1310.2085=' + (pageTiming.domInteractive - pageTiming.responseStart),

            // [dom.init] Время выполнения DOMContentLoaded
            '1310.1309=' + (domContentLoadedEventEnd - domContentLoadedEventStart),

            // [dom.loaded] Время начала DOMContentLoaded от первого байта
            '1310.1007=' + (domContentLoadedEventStart - pageTiming.responseStart),

            // [device_memory] Объём памяти устройства, см. https://github.com/w3c/device-memory
            navigator.deviceMemory && '3140=' + navigator.deviceMemory,

            // [hardware_concurrency] Количество ядер, см. https://clck.ru/DYqo9
            navigator.hardwareConcurrency && '3141=' + navigator.hardwareConcurrency
        ]);

        /*
         * Дополнить список переменных закодированными таймингами. Выбираются только тайминги, указанные в списке
         * COMMON_TIMING_PROPS_MAP, имеющие truthy-значение (всё, кроме нуля, null, undefined, пустой строки)
         * и пересчитываются относительно navStart.
         *
         * В PerformanceNavigationTiming поля с нулевыми значениями показывают, что данный счётчик не сработал
         * (редиректа не было, DCL ещё не наступил и т.п.) - поэтому нули надо отфильтровать.
         *
         * @see https://developer.mozilla.org/en-US/docs/Web/API/PerformanceNavigationTiming
         */
        Object.keys(COMMON_TIMING_PROPS_MAP).forEach(function(name) {
            if ((name in pageTiming) && pageTiming[name]) {
                vars.push(COMMON_TIMING_PROPS_MAP[name] + '=' + normalize(pageTiming[name], navStart));
            }
        });

        if (rum.vsStart) {
            // [visibility] См. MDN: https://clck.ru/9edpN
            // 2771 – invalid
            vars.push('1484=' + (VISIBILITY_STATES[rum.vsStart] || 2771));

            // [visibility.change] Видимость менялась на протяжение загрузки
            if (rum.vsChanged) {
                vars.push('1484.719=1');
            }
        } else {
            // Для АБ отправляем счетчик видимости для браузеров
            // без поддержки visibilityState
            vars.push('1484=' + VISIBILITY_STATES['visible']);
        }

        if (navigation) {
            if (navigation.redirectCount) {
                // [redirect.count] Количество редиректов
                vars.push('1384.1385=' + navigation.redirectCount);
            }

            // [nav.type] См. MDN: https://clck.ru/9edos
            // 1 – Навигация через перезагрузку страницы или location.reload()
            // 2 – Навигация по истории (назад, вперёд)
            if (navigation.type === 1 || navigation.type === 2) {
                vars.push('770.76=' + navigation.type);
            }
        }

        pushConnectionTypeTo(vars);

        sendCounter(PATH_TECH_TIMING, vars);
    }

    var paintEventsSent = {};
    var paintEventsSentCount = 0;

    /**
     * Отправить данные об отрисовках частей страницы.
     */
    function sendPaintTiming() {
        if (
            !resourceTimingSupported ||
            !rum.getSetting('forcePaintTimeSending') && rum.isVisibilityChanged()
        ) {
            return;
        }

        var paintEvents = perf.getEntriesByType('paint');

        for (var i = 0; i < paintEvents.length; i++) {
            var event = paintEvents[i];
            var paintType = PAINT_TYPES[event.name];

            if (paintType && !paintEventsSent[event.name]) {
                paintEventsSent[event.name] = true;
                paintEventsSentCount++;
                sendTimeMark('1926.' + paintType, event.startTime);
            }
        }

        // На DomContentLoaded отрисовка может не произойти (например в случае полностью клиентского рендеринга)
        // В этом случае нужно дождаться события отрисовки и попробовать отправить метрики ещё раз
        if (paintEventsSentCount < PAINT_EVENTS_COUNT) {
            try {
                (new PerfObserver(function(list, observer) {
                    sendPaintTiming();

                    // https://st.yandex-team.ru/SERP-91162
                    if (observer) {
                        observer.disconnect();
                    }
                })).observe({entryTypes: ['paint']});
            } catch (e) {
            }
        }
    }

    /**
     * Отправить счетчик с временной меткой.
     *
     * @param {String} counterId - Код BlockStat. ID метки времени
     * @param {Number} [time] - Время. Если не передано, вычисляется время от начала навигации до момента вызова функции
     * @param {Boolean} [addPerfMark=true] - Добавлять метку в User Timing или нет, true по умолчанию
     * @param {Object} [params=null] - Кастомные параметры счетчика. Сюда же можно передать инстанс подстраницы
     */
    function sendTimeMark(counterId, time, addPerfMark, params) {
        if (time === undefined) {
            time = getTime();
        }

        if (addPerfMark === undefined || addPerfMark === true) {
            rum.mark(counterId, time);
        }

        var vars = getMarkCommonVars(counterId);

        vars.push('207=' + normalize(time)); // time (время от navigation start)

        if (!addCustomParamsToVars(vars, params)) {
            return;
        }

        sendCounter(PATH_TIME, vars);

        timeMarks[counterId] = timeMarks[counterId] || [];
        timeMarks[counterId].push(time);

        var listeners = rum._markListeners[counterId];

        if (listeners && listeners.length) {
            listeners.forEach(function(cb) {
                cb(time);
            });
        }
    }

    /**
     * Добавить кастомные параметры в метрики запроса.
     *
     * @param {Array} vars - Список с метриками
     * @param {Object} [params=null] - Кастомные параметры счетчика
     */
    function addCustomParamsToVars(vars, params) {
        if (params) {
            // Если в отложенном запуске решили, что счетчик уже не нужен
            if (params.isCanceled && params.isCanceled()) {
                return false;
            }

            var key2IdxMap = vars.reduce(function(acc, el, idx) {
                if (typeof el === 'string') {
                    var key = el.split('=')[0];
                    acc[key] = idx;
                }
                return acc;
            }, {});

            Object.keys(params).forEach(function(paramName) {
                if (typeof params[paramName] !== 'function') {
                    // перезаписываем, если уже есть такой параметр
                    var idx = key2IdxMap[paramName],
                        value = paramName + '=' + params[paramName];

                    if (idx === undefined) {
                        vars.push(value);
                    } else {
                        vars[idx] = value;
                    }
                }
            });
        }
        return true;
    }

    /**
     * Получить отправленные на данный момент метки времени.
     *
     * @returns {Object}
     */
    rum.getTimeMarks = function() {
        return timeMarks;
    };

    /**
     * Отмечает время окончания расчёта дельты и отправляет дельту.
     *
     * @param {String} counterId - Код BlockStat. ID метки времени
     * @param {Object} [vars] - Объект с дополнительными переменными
     */
    function timeEnd(counterId, vars) {
        var deltaTimes = deltaMarks[counterId];

        if (!deltaTimes || deltaTimes.length === 0) {
            return;
        }

        deltaTimes.push(getTime(), vars);

        sendDelta(counterId);
    }

    /**
     * Отправить счетчик с дельтой времени.
     *
     * @param {String} counterId - Код BlockStat. ID метки времени
     * @param {Number} [delta] - Точное значение дельты
     * @param {Object} [subPage] - Инстанс подстраницы
     */
    function sendDelta(counterId, delta, subPage) {
        var deltaTimes = deltaMarks[counterId],
            deltaStart,
            deltaEnd,
            deltaVars;

        // используем явно заданное значение из дельты
        if (typeof delta !== 'undefined') {
            deltaEnd = rum.getTime();
            deltaStart = deltaEnd - delta;
        } else if (deltaTimes) {
            deltaStart = deltaTimes[0];
            deltaEnd = deltaTimes[1];
            deltaVars = deltaTimes[2];
        }

        if (deltaStart === undefined || deltaEnd === undefined) {
            return;
        }

        var vars = getMarkCommonVars(counterId);

        vars.push(
            '207.2154=' + normalize(deltaStart), // time.start
            '207.1428=' + normalize(deltaEnd), // time.end
            '2877=' + normalize(deltaEnd - deltaStart) // delta
        );

        if (addCustomParamsToVars(vars, subPage) && addCustomParamsToVars(vars, deltaVars)) {
            sendCounter(PATH_DELTA, vars);
            delete deltaMarks[counterId];
        }
    }

    /**
     * Отправить счётчик с временными метриками ресурса.
     *
     * @param {String|Number} counterId - Код BlockStat. ID метки времени
     * @param {String} url
     */
    function sendResTiming(counterId, url) {
        getResourceTimings(url, function(resTimings) {
            if (resTimings) {
                var vars = getMarkCommonVars(counterId);

                if (rum.getSetting('sendUrlInResTiming')) {
                    vars.push('13=' + encodeURIComponent(url));
                }

                pushTimingTo(vars, resTimings[0]);
                sendCounter(PATH_RESOURCE_TIMING, vars);
            }
        });
    }

    var periodicCounterIntervalId;
    rum._periodicTasks = [];

    /**
     * Стартовать таймер периодических статистик страницы
     */
    function initPeriodicStats() {
        var interval = rum.getSetting('periodicStatsIntervalMs');
        if (!interval && interval !== null) {
            interval = 15000;
        }

        if (interval) {
            periodicCounterIntervalId = setInterval(periodicStatsTick, interval);
        }
        addEventListener('beforeunload', periodicStatsTick);
    }

    /**
     * Обработчик периодических задач сбора статистики
     */
    function periodicStatsTick() {
        var used = false;

        rum._periodicTasks.forEach(function(task) {
            if (task()) {
                used = true;
            }
        });

        if (!used) {
            clearInterval(periodicCounterIntervalId);
        }
    }

    var trafficCounterSentCount = 0;

    /**
     * Стартовать отправку данных о потреблённом страницей трафике
     */
    function initTrafficStats() {
        if (!PerfObserver) {
            return;
        }

        var doc = perf.getEntriesByType('navigation');
        writeTrafficData(doc);

        var resources = perf.getEntriesByType('resource');
        writeTrafficData(resources);

        try {
            (new PerfObserver(function(entryList) {
                writeTrafficData(entryList.getEntries());
            })).observe({ entryTypes: ['resource', 'navigation'] });
        } catch (e) {}

        rum._periodicTasks.push(sendTrafficData);
    }

    function writeTrafficData(entries) {
        if (!entries || !entries.length) {
            return;
        }

        var db = trafficDataBuffer;

        for (var i = 0; i < entries.length; i++) {
            var entryData = getTrafficEntryData(entries[i]);
            if (!entryData) {
                continue;
            }

            var groupKey = entryData.domain + '-' + entryData.extension;

            var buf = db[groupKey] = db[groupKey] || {
                count: 0,
                size: 0
            };
            buf.count++;
            buf.size += entryData.size;
        }
    }

    function getTrafficEntryData(entry) {
        var size = entry.transferSize;
        if (size == null) {
            return;
        }

        normalizeAnchor.href = entry.name;

        var pathname = normalizeAnchor.pathname;
        if (pathname.indexOf('/clck') === 0) {
            return;
        }

        var lastDotPath = pathname.lastIndexOf('.');
        var extension = '';
        if (lastDotPath != -1 && pathname.lastIndexOf('/') < lastDotPath && pathname.length - lastDotPath <= 5) {
            extension = pathname.slice(lastDotPath + 1);
        }

        return {
            size: size,
            domain: normalizeAnchor.hostname,
            extension: extension
        };
    }

    function sendTrafficData() {
        var maxCounters = rum.getSetting('maxTrafficCounters') || 250;
        if (trafficCounterSentCount >= maxCounters) {
            return false;
        }

        var groupKeys = Object.keys(trafficDataBuffer);
        var value = '';

        for (var i = 0; i < groupKeys.length; i++) {
            var groupKey = groupKeys[i];
            var groupData = trafficDataBuffer[groupKey];
            value += encodeURIComponent(groupKey) + '!' + groupData.count + '!' + groupData.size + ';';
        }

        if (value.length) {
            trafficCounterSentCount++;
            var currentVars = commonVars.concat(['d=' + value, 't=' + normalize(getTime())]);
            sendCounter(TECH_PERF_TRAFFIC, currentVars);
        }

        trafficDataBuffer = {};
        return trafficCounterSentCount < maxCounters;
    }

    /**
     * Отправить метрику First Input Delay
     * See: https://web.dev/fid/
     */
    function sendFirstInputDelay() {
        if (!PerfObserver) {
            return;
        }

        try {
            (new PerfObserver(function(entryList, observer) {
                var entry = entryList.getEntries()[0];
                if (!entry) {
                    return;
                }

                var fid = entry.processingStart - entry.startTime;
                sendDelta('first-input', fid);

                observer.disconnect();
            })).observe({ type: 'first-input', buffered: true });
        } catch (e) {}
    }

    var cumulativeLayoutShiftScore;

    /**
     * Посчитать Cumulative Layout Shift (CLS)
     * See: https://web.dev/cls/
     */
    function calcLayoutShiftScore() {
        if (!PerfObserver) {
            return;
        }

        var po = new PerfObserver(function(entryList) {
            var entries = entryList.getEntries();

            if (cumulativeLayoutShiftScore == null) {
                cumulativeLayoutShiftScore = 0;
            }

            for (var i = 0; i < entries.length; i++) {
                var entry = entries[i];
                if (!entry.hadRecentInput) {
                    cumulativeLayoutShiftScore += entry.value;
                }
            }
        });

        try {
            po.observe({ type: 'layout-shift', buffered: true });
        } catch (e) {}

        addEventListener('visibilitychange', function onVisChange() {
            if (document.visibilityState !== 'hidden') {
                return;
            }

            removeEventListener('visibilitychange', onVisChange);

            try {
                if (typeof po.takeRecords === 'function') {
                    po.takeRecords(); // очистить буфер
                }
                po.disconnect();
            } catch (e) {}

            finalizeLayoutShiftScore();
        });

        addEventListener('beforeunload', finalizeLayoutShiftScore);
    }

    function finalizeLayoutShiftScore() {
        if (cumulativeLayoutShiftScore == null) {
            return;
        }

        var score = Math.round(cumulativeLayoutShiftScore * 1e6) / 1e6;
        sendCounter(TECH_PERF_CLS, commonVars.concat('s=' + score));

        cumulativeLayoutShiftScore = null;
    }

    var largestContentfulPaint = null;
    var largestLoadingElementPaintSent = false;

    /**
     * Посчитать Largest Contentful Paint (LCP)
     * Также считается Largest Loading Element Paint
     * See: https://web.dev/lcp/
     */
    function calcLargestContentfulPaint() {
        if (!PerfObserver || !rum.getSetting('forcePaintTimeSending') && rum.isVisibilityChanged()) {
            return;
        }

        var po = new PerfObserver(function(entryList) {
            var entries = entryList.getEntries();

            for (var i = 0; i < entries.length; i++) {
                var entry = entries[i];
                largestContentfulPaint = entry.renderTime || entry.loadTime;
            }

            if (!largestLoadingElementPaintSent) {
                sendTimeMark('largest-loading-elem-paint', largestContentfulPaint);
                largestLoadingElementPaintSent = true;
            }
        });

        try {
            po.observe({ type: 'largest-contentful-paint', buffered: true });
        } catch (e) {}

        addEventListener('visibilitychange', function onVisChange() {
            if (document.visibilityState !== 'hidden') {
                return;
            }

            removeEventListener('visibilitychange', onVisChange);

            try {
                if (typeof po.takeRecords === 'function') {
                    po.takeRecords(); // очистить буфер
                }
                po.disconnect();
            } catch (e) {}

            finalizeLargestContentfulPaint();
        });

        addEventListener('beforeunload', finalizeLargestContentfulPaint);
    }

    function finalizeLargestContentfulPaint() {
        if (largestContentfulPaint == null) {
            return;
        }

        sendTimeMark('largest-contentful-paint', largestContentfulPaint);

        largestContentfulPaint = null;
    }

    /**
     * Отправить метрики Element Timing API
     * See: https://wicg.github.io/element-timing/
     */
    var sendElementTiming = PerfObserver ? function() {
        if (!rum.getSetting('forcePaintTimeSending') && rum.isVisibilityChanged()) {
            return;
        }

        try {
            (new PerfObserver(function(entryList) {
                var entries = entryList.getEntries();

                for (var i = 0; i < entries.length; i++) {
                    var entry = entries[i];
                    sendTimeMark('element-timing.' + entry.identifier, entry.startTime); // startTime = renderTime||loadTime
                }
            })).observe({ type: 'element', buffered: true });
        } catch (e) {}
    } :
    function() {};

    /**
     * Ожидает завершения long task'ов и интервала в LONG_TASK_TIMEOUT мс, начиная с текущего момента.
     *
     * @param {String} [name='2795'] - название события
     * @param {Number} [start] - начало подсчетов относительно жизненного цикла страницы
     * @param {Object} [subPage] - инстанс подстраницы
     */
    function checkLongTasks(name, start, subPage) {
        if (!rum._tti) {
            return;
        }

        var currTime = getTime();

        var sendLongTasks = function(name, time, longTasks) {
            var markParams = {
                '2796.2797': getLongtasksStringValue(longTasks, start), // long-task.value
                '689.2322': normalize(currTime) // action.start_time
            };

            if (subPage) {
                Object.keys(subPage).forEach(function(paramName) {
                    markParams[paramName] = subPage[paramName];
                });
            }

            sendTimeMark(name || '2795', time, true, markParams);

            rum._tti.fired = true;
        };

        function check() {
            var maybeFCI = start || currTime;
            var now = getTime();
            var longTasks = rum._tti.events || [];
            var longTasksLength = longTasks.length;
            var lastLongTask;

            if (longTasksLength !== 0) {
                lastLongTask = longTasks[longTasksLength - 1];
                maybeFCI = Math.max(maybeFCI, Math.floor(lastLongTask.startTime + lastLongTask.duration));
            }

            if (now - maybeFCI >= LONG_TASK_TIMEOUT) {
                sendLongTasks(name, maybeFCI, longTasks); // tti
            } else if (now - maybeFCI >= LONG_TASK_LIMIT) {
                sendLongTasks(name, (start || currTime) + LONG_TASK_LIMIT, longTasks);  // tti
            } else {
                setTimeout(check, 1000);
            }
        }

        check();
    }

    // Вынесем наружу для того, чтобы можно было засекать начиная с любого момента
    rum.sendTTI = checkLongTasks;

    /**
     * Отправляет событие об отрисовке основного содержимого.
     *
     * @param {Number} [time] - время относительно жизненного цикла страницы
     */
    rum.sendHeroElement = function(time) {
        sendTimeMark('2876', time);
    };

    rum._subpages = {};

    /**
     * Создает инстанс подстраницы, чтобы отправлять счетчики в отдельном неймспейсе.
     *
     * @param {String} pageName - имя подстраницы
     * @param {String} [start] - время создания
     */
    rum.makeSubPage = function(pageName, start) {
        var index = rum._subpages[pageName];

        if (typeof index === 'undefined') {
            rum._subpages[pageName] = index = 0;
        } else {
            rum._subpages[pageName] = ++index;
        }

        var canceled = false;

        return {
            '689.2322': normalize(typeof start !== 'undefined' ? start : getTime()), // action.start_time
            2924: pageName, // bs: subpage_name
            2925: index, // bs: subpage_index
            isCanceled: function() {
                return canceled;
            },
            cancel: function() {
                canceled = true;
            }
        };
    };

    //
    // Хелперы
    //
    function pushConnectionTypeTo(vars) {
        if (connection) {
            vars.push(
                // [connection_type] Тип соединения. MDN: https://clck.ru/AASTf
                // 2771 – invalid
                '2437=' + (CONNECTION_TYPES[connection.type] || 2771),

                // [downlink_max] Максимальная скорость соединения. MDN: https://clck.ru/AASYS
                connection.downlinkMax !== undefined && ('2439=' + connection.downlinkMax),
                // [effective_type] Оценка реального качества соединения. MDN: https://clck.ru/C5hCb
                connection.effectiveType && ('2870=' + connection.effectiveType),
                // Характеристики сети
                connection.rtt !== undefined && ('rtt=' + connection.rtt),
                connection.downlink !== undefined && ('dwl=' + connection.downlink)
            );
        }
    }

    function getResourceTimings(url, cb) {
        if (!resourceTimingSupported) {
            return cb(null);
        }

        normalizeLink.href = url;

        var normalizedUrl = normalizeLink.href;
        var pollingIteration = 0;
        var timeBeforeNextPolling = 100;

        function tryToGetResTiming() {
            var resTimings = perf.getEntriesByName(normalizedUrl);

            if (resTimings.length) {
                return cb(resTimings);
            }

            if (pollingIteration++ < POLLING_ITERATIONS_LIMIT) {
                setTimeout(tryToGetResTiming, timeBeforeNextPolling);
                timeBeforeNextPolling += timeBeforeNextPolling;
            } else {
                cb(null);
            }
        }

        setTimeout(tryToGetResTiming, 0);
    }

    /**
     * @typedef {Object} LongTask
     * @property {String} name
     * @property {Number} startTime
     * @property {Number} duration
     */

    /**
     * Получить строковое значение лонгтасков для отправки.
     *
     * @param {Array<LongTask>} longtasks - массив лонгтасков
     * @param {Number} [start] - начало подсчетов относительно жизненного цикла страницы
     * @return {String}
     * @private
     */
    function getLongtasksStringValue(longtasks, start) {
        longtasks = longtasks || [];
        start = start || 0;

        return longtasks.filter(function(longtask) {
            // учитываем последний начавшийся longtask
            // есть погрешность Ya.Rum.getTime() и longtask.startTime в единицы мс
            return longtask.startTime - start >= -50;
        }).map(function(longtask) {
            // Возможные имена тасков:
            // https://w3c.github.io/longtasks/#sec-PointingToCulprit пункт 3.3
            // Имена тасков сокращаем до первых букв и делаем уникальными, добавляя время таска
            // Например 'same-origin-descendant' -> 'sod-840-907'
            var name = longtask.name ? longtask.name.split('-').map(function(el) {
                return el[0];
            }).join('') : 'u';

            var start = Math.floor(longtask.startTime),
                end = Math.floor(start + longtask.duration);

            return name + '-' + start + '-' + end;
        }).join('.');
    }

    rum._getLongtasksStringValue = getLongtasksStringValue;

    function getMarkCommonVars(counterId) {
        return markCommonVars.concat([
            '1701=' + counterId, // id – идентификатор счётчика
            rum.ajaxStart && ('1201.2154=' + normalize(rum.ajaxStart)), // ajax.start
            rum.ajaxComplete && ('1201.2052=' + normalize(rum.ajaxComplete)) // ajax.success
        ]);
    }

    /**
     * Дополнить список переменных закодированными таймингами. Выбираются только тайминги, указанные в списке
     * COMMON_TIMING_PROPS_MAP, имеющие truthy-значение или ноль.
     *
     * Используется для извлечения значений PerformanceResourceTiming, там нули вполне валидны и стабильно
     * воспроизводятся в Hermione-тестах в web4, и на текущий момент некоторые из таких полей помечены в тестах
     * как обязательные.
     *
     * @see https://developer.mozilla.org/en-US/docs/Web/API/PerformanceResourceTiming
     *
     * @param {Array<String>} vars - список переменных, который надо дополнить
     * @param {Object} timing - объект с таймингами
     */
    function pushTimingTo(vars, timing) {
        Object.keys(COMMON_TIMING_PROPS_MAP).forEach(function(name) {
            if (name in timing) {
                var time = timing[name];
                if (time || time === 0) {
                    vars.push(COMMON_TIMING_PROPS_MAP[name] + '=' + normalize(time));
                }
            }
        });
    }

    /**
     * Подготовить значение к отправке: строку - закодировать, число - пересчитать относительно заданного смещения и
     * округлить до трёх знаков после запятой.
     *
     * @param {String|Number} x
     * @param {Number} [offset]
     * @returns {String|Number}
     */
    function normalize(x, offset) {
        return typeof x === 'string' ? encodeURIComponent(x) : Math.round((x - (offset || 0)) * 1000) / 1000;
    }

    /**
     * Отправляет счётчик
     *
     * @param {String} path
     * @param {Array<String>} varsArr
     */
    function sendCounter(path, varsArr) {
        var cdnLocation = encodeURIComponent(window.YaStaticRegion || 'unknown');

        varsArr.push('-cdn=' + cdnLocation);

        var vars = varsArr.filter(Boolean).join(',');

        rum.send(null, path, vars);
    }

    // Предоставляем внутренние функции опциональным модулям
    rum.getResourceTimings = getResourceTimings;
    rum.pushConnectionTypeTo = pushConnectionTypeTo;
    rum.pushTimingTo = pushTimingTo;
    rum.normalize = normalize;
    rum.sendCounter = sendCounter;
    rum.sendDelta = sendDelta;
    rum.sendTrafficData = sendTrafficData;
    rum.finalizeLayoutShiftScore = finalizeLayoutShiftScore;
    rum.finalizeLargestContentfulPaint = finalizeLargestContentfulPaint;
    rum.onReady = onReady;
})(Ya.Rum);
