(function(rum) {
    var TECH_PERF_LONG_TASK = '690.2096.2796'; // tech.perf.long-task
    var TECH_PERF_SCROLL = '690.2096.768'; // tech.perf.scroll
    var TECH_PERF_SCROLL_LATENCY = '690.2096.768.2373'; // tech.perf.scroll.latency

    var longTaskCounter = 0;

    /**
     * Отправляем собранные long task'и
     *
     * @returns {Boolean} Нужно ли продолжать периодический сбор статистики
     */
    function sendLongTaskData() {
        if (!rum._tti.fired) {
            return true;
        }

        var maxCounters = rum.getSetting('maxLongTaskCounters') || 10;
        if (++longTaskCounter > maxCounters) {
            return false;
        }

        if (rum._tti.eventsAfterTTI.length) {
            var vars = rum._getCommonVars().slice();
            vars.push('2796.2797=' + rum._getLongtasksStringValue(rum._tti.eventsAfterTTI));
            vars.push('689.2322=' + rum.normalize(rum.getTime()));
            rum.sendCounter(TECH_PERF_LONG_TASK, vars);
            rum._tti.eventsAfterTTI = [];
        }

        return true;
    }

    var scrollTimes = [];

    /**
     * Стартует сбор статистики скролла
     */
    function initScrollStats() {
        var scrollStarted = false;
        var scrollStartTime = rum.getTime();
        var prevScroll;
        var SCROLL_TIMEOUT = 500;
        var timeout;

        addEventListener('scroll', onScroll);

        function onScroll() {
            var now = rum.getTime();

            if (!scrollStarted) {
                scrollStarted = true;
                scrollStartTime = rum.getTime();
            } else {
                scrollTimes.push(now - prevScroll);
            }

            prevScroll = now;

            clearTimeout(timeout);
            timeout = setTimeout(function() {
                scrollStarted = false;
            }, SCROLL_TIMEOUT);
        }
    }

    /**
     * Считает среднеквадратическое
     *
     * @param {Array<Number>} times
     * @returns {Number}
     */
    function calcScrollValue(times) {
        var sum = 0;
        for (var i = 0; i < times.length; ++i) {
            sum += times[i] * times[i];
        }
        return Math.sqrt(sum / times.length);
    }

    /**
     * Отправляет данные по скроллу
     *
     * @returns {Boolean} Нужно ли продолжать периодический сбор статистики
     */
    function sendScrollData() {
        if (scrollTimes.length) {
            var currentVars = rum._getCommonVars().concat(['d=' + calcScrollValue(scrollTimes)]);
            rum.sendCounter(TECH_PERF_SCROLL, currentVars);
            scrollTimes = [];
        }

        return true;
    }

    var scrollLatencyItems = [];

    /**
     * Инициализирует счётчик задержки скролла (YaBrowser 20.2+)
     * @returns {Boolean}
     */
    function initScrollLatencyStats() {
        try {
            if (!window.PerformanceScrollUpdateLatency) {
                return false;
            }

            var observer = new PerformanceObserver(function (list) {
                list.getEntries().forEach(function (entry) {
                    scrollLatencyItems.push(entry.duration);
                });
            });
            observer.observe({entryTypes: ['scrollUpdateLatency']});
        } catch (err) {
            return false;
        }

        return true;
    }

    /**
     * Отправляет данные задержки скролла
     *
     * @returns {Boolean} Нужно ли продолжать периодический сбор статистики
     */
    function sendScrollLatencyData() {
        if (scrollLatencyItems.length) {
            var currentVars = rum._getCommonVars().concat(['d=' + calcScrollValue(scrollLatencyItems)]);
            rum.sendCounter(TECH_PERF_SCROLL_LATENCY, currentVars);
            scrollLatencyItems = [];
        }

        return true;
    }

    if (rum.enabled) {
        if (rum.getSetting('longTaskMetric') && rum._tti) {
            rum._periodicTasks.push(sendLongTaskData);
        }

        if (rum.getSetting('scrollMetric')) {
            initScrollStats();
            rum._periodicTasks.push(sendScrollData);
        }

        if (rum.getSetting('scrollLatencyMetric')) {
            if (initScrollLatencyStats()) {
                rum._periodicTasks.push(sendScrollLatencyData);
            }
        }
    }
})(Ya.Rum);
