declare namespace Ya {
    namespace Rum {
        /**
         * флаг для проверки работоспособности счетчика
         */
        const enabled: boolean;

        /**
         * Видимость документа в начале загрузки
         */
        const vsStart: boolean;

        /**
         * Менялась ли видимость документа document.visibilityState в процессе загрузки
         */
        const vsChanged: boolean;

        /**
         * список временных меток для дальнейшей отправки
         */
        const _defTimes: number[];

        /**
         * список меток ресурсов для дальнейшей отправки
         */
        const _defRes: number[];

        /**
         * список рассчитанных с помощью time()..timeEnd() дельт
         */
        const _deltaMarks: Record<string, number>;

        /**
         * список обработчиков, подписанных на срабатывание меток
         */
        const _markListeners: Record<string, Function>;

        /**
         * Настройки RUM
         */
        const _settings: RUMSettings;

        /**
         * Произвольные параметры, пробрасываемые в логи.
         *
         * @example
         * ```ts
         * {
         *     "143": "28.1786",
         *     "1042": "Example%20User%20Agent%201.0"
         * }
         * ```
         */
        const _vars: Record<string, string>;

        /**
         * Параметры RUM счетчика
         */
        interface RUMSettings {
            /**
             * использовать beacon API или нет
             */
            beacon?: boolean,

            /**
             * хост для отправки счетчика
             */
            clck?: string,

            /**
             * test-id's или другие идентификаторы выборок, в которые попал показ
             */
            slots?: string[],

            /**
             * уникальный id показа
             */
            reqid?: string,

            /**
             * Всегда отправлять отрисовки (например, при пререндере)
             */
            forcePaintTimeSending?: boolean,

            /**
             * Отправлять клиентский UA
             */
            sendClientUa?: boolean;

            /**
             * Отправлять before_bem-init
             */
            sendBeforeBemInited?: boolean;

            /**
             * Отправлять URL в Resource Timing
             */
            sendUrlInResTiming?: boolean;

            /**
             * Автоматически отправлять Resource Timing для размеченных
             */
            sendAutoResTiming?: boolean;

            /**
             * Автоматически отправлять Element Timing для размеченных
             */
            sendAutoElementTiming?: boolean;

            /**
             * Отправлять первый rAF в методе sendRaf
             */
            sendFirstRaf?: boolean;

            /**
             * Отключить отправку счетчиков по `onload`
             */
            disableOnLoadTasks?: boolean;

            /**
             * Интервал периодической отправки счетчиков о потребленном трафике в миллисекундах
             * Значение `null` отключает отправку
             *
             * @defaultValue 15000
             */
            periodicStatsIntervalMs?: number | null;

            /**
             * Максимальное количество отправленных счётчиков трафика
             *
             * @defaultValue 250
             */
            maxTrafficCounters?: number;

            /**
             * Максимальное количество отправленных счётчиков долгих операций
             *
             * @defaultValue 10
             */
            maxLongTaskCounters?: number;

            /**
             * Общий счетчик метрик скролла
             */
            scrollMetric?: boolean;

            /**
             * Счетчик метрик скролла в ЯБро
             */
            scrollLatencyMetric?: boolean;

            /**
             * Список экспериментов
             */
            experiments?: string[];
        }

        /**
         * Инициализация интерфейса.
         */
        function init(settings: RUMSettings, vars: Record<string, string>): void;

        /**
         * Получить время от начала навигации.
         */
        function getTime(): number;

        /**
         * Записывает время старта расчёта дельты.
         *
         * @param counterId - Код BlockStat. ID метки времени
         */
        function time(counterId: string): void;

        /**
         * Записывает время окончания расчёта дельты.
         * При загрузке RUM implementation записанные метрики будут отправлены на бэкэнд.
         *
         * @param counterId - Код BlockStat. ID метки времени
         * @param vars - Объект с дополнительными переменными
         */
        function timeEnd(counterId: string, vars?: object): void;

        /**
         * Отправить счетчик с временной меткой.
         *
         * @param counterId Код BlockStat. ID метки времени
         * @param time - Время. Если не передано, вычисляется время
         *     от начала навигации до момента вызова функции
         * @param addPerfMark - Добавлять метку в User Timing или нет, true по умолчанию
         * @param params - Кастомные параметры счетчика. Сюда же можно передать инстанс подстраницы
         */
        function sendTimeMark(
            counterId: string,
            time?: number,
            addPerfMark?: boolean,
            params?: { [ s: string ]: string; }
        ): void;

        /**
         * Заглушка отправки временных меток ресурса,
         * откладывает отправку до загрузки основного кода.
         */
        function sendResTiming(counterId: string | number, url: string): void;

        /**
         * Отправить тайминги requestAnimationFrame.
         *
         * @param position – blockStat-код позиции счётчика, например 85 – p0
         */
        function sendRaf(position: string | number): void;

        /**
         * Проверяет, изменялась ли видимость страницы при загрузке.
         */
        function isVisibilityChanged(): boolean;

        /**
         * Поставить метку времени в User Timing https://www.w3.org/TR/user-timing/#dom-performance-mark
         * В браузерах, не поддерживающих window.performance.mark(), заменяется ничего не делающей заглушкой.
         *
         * @param name - Код BlockStat. ID метки времени
         * @param time - Время. Опционально, нужно для сверки со временем, которое ставит
         *     само User Timing API, или для меток, которые ставятся "задним числом" (ttfb, ttfp etc.)
         */
        function mark(name: string, time?: number): void;

        /**
         * Получить значение настройки.
         */
        function getSetting(settingName: string): string | null;

        /**
         * Подписаться на срабатывание временной метки
         */
        function on(counterId: string, cb: Function): void;

        /**
         * Отправить данные о трафике
         */
        function sendTrafficData(): void;

        /**
         * Отправить текущее значение CLS и начать считать заново
         */
        function finalizeLayoutShiftScore(): void;

        /**
         * Отправить текущее значение LCP и начать искать заново
         */
        function finalizeLargestContentfulPaint(): void;
    }
}
