/**
 * @file
 * Интерфейс счётчиков скорости. Этот код должен быть помещён раньше элемента implementation.
 *
 * Код оптимизирован для минимального размера после uglify + gzip.
 */
(function(window, undefined) {
    window.Ya = window.Ya || {};

    /**
     * @typedef {number|string} BlockStatID – ключ из словаря https://stat.yandex-team.ru/DictionaryEditor/BlockStat
     */

    if (Ya.Rum) {
        throw new Error('Rum: interface is already defined');
    }

    var perf = window.performance;
    var navStart = (perf && perf.timing && perf.timing.navigationStart) || Ya.startPageLoad || +new Date();
    var requestAnimationFrame = window.requestAnimationFrame;

    Ya.Rum = {
        // флаг для проверки работоспособности счетчика
        enabled: !!perf,

        // Видимость документа в начале загрузки
        vsStart: document.visibilityState,

        // Менялась ли видимость документа document.visibilityState в процессе загрузки
        vsChanged: false,

        // список временных меток для дальнейшей отправки
        _defTimes: [],

        // список меток ресурсов для дальнейшей отправки
        _defRes: [],

        // список рассчитанных с помощью time()..timeEnd() дельт
        _deltaMarks: {},

        // список обработчиков, подписанных на срабатывание меток
        _markListeners: {},

        /**
         * Настройки RUM
         *
         * @typedef {Object} RUMSettings
         */
        _settings: {},

        /**
         * Произвольные параметры, пробрасываемые в логи.
         *
         * @typedef {Object<BlockStatID, String|BlockStatID>} Vars – Объект с параметрами
         * @example
         * {
         *     "143": "28.1786",
         *     "1042": "Example%20User%20Agent%201.0"
         * }
         */
        _vars: {},

        /**
         * Инициализация интерфейса.
         *
         * @param {RUMSettings} settings – Настройки RUM
         * @param {Vars} vars
         */
        init: function(settings, vars) {
            this._settings = settings;
            this._vars = vars;
        },

        /**
         * Получить время от начала навигации.
         *
         * @returns {Number}
         */
        getTime: (perf && perf.now) ? function() {
            return perf.now();
        } : Date.now ? function() {
            return Date.now() - navStart;
        } : function() {
            return new Date - navStart;
        },

        /**
         * Записывает время старта расчёта дельты.
         *
         * @param {String} counterId - Код BlockStat. ID метки времени
         */
        time: function(counterId) {
            this._deltaMarks[counterId] = [this.getTime()];
        },

        /**
         * Записывает время окончания расчёта дельты.
         * При загрузке RUM implementation записанные метрики будут отправлены на бэкэнд.
         *
         * @param {String} counterId - Код BlockStat. ID метки времени
         * @param {Object} [vars] - Объект с дополнительными переменными
         */
        timeEnd: function(counterId, vars) {
            var deltaMarks = this._deltaMarks,
                deltaTimes = deltaMarks[counterId];

            if (!deltaTimes || deltaTimes.length === 0) {
                return;
            }

            deltaTimes.push(this.getTime(), vars);
        },

        /**
         * Отправить счетчик с временной меткой.
         *
         * @param {String} counterId Код BlockStat. ID метки времени
         * @param {Number} [time] - Время. Если не передано, вычисляется время
         *     от начала навигации до момента вызова функции
         * @param {Boolean} [addPerfMark=true] - Добавлять метку в User Timing или нет, true по умолчанию
         * @param {Object} [params=null] - Кастомные параметры счетчика. Сюда же можно передать инстанс подстраницы
         */
        sendTimeMark: function(counterId, time, addPerfMark, params) {
            if (time === undefined) {
                time = this.getTime();
            }
            this._defTimes.push([counterId, time, params]);
            this.mark(counterId, time);
        },

        /**
         * Заглушка отправки временных меток ресурса,
         * откладывает отправку до загрузки основного кода.
         *
         * @param {String|Number} counterId
         * @param {String} url
         */
        sendResTiming: function(counterId, url) {
            this._defRes.push([counterId, url]);
        },

        /**
         * Отправить тайминги requestAnimationFrame.
         *
         * @param {String|Number} position – blockStat-код позиции счётчика, например 85 – p0
         */
        sendRaf: function(position) {
            var forcePaintTimeSending = this.getSetting('forcePaintTimeSending');

            if (!requestAnimationFrame || !forcePaintTimeSending && this.isVisibilityChanged()) {
                return;
            }

            var that = this,
                prefix = '2616.' + position; // raf_metric.p0

            requestAnimationFrame(function() {
                if (!forcePaintTimeSending && that.isVisibilityChanged()) {
                    return;
                }

                if (that.getSetting('sendFirstRaf')) {
                    that.sendTimeMark(prefix + '.205'); // raf_metric.p0.first
                }

                requestAnimationFrame(function() {
                    if (!forcePaintTimeSending && that.isVisibilityChanged()) {
                        return;
                    }
                    that.sendTimeMark(prefix + '.1928'); // raf_metric.p0.last
                });
            });
        },

        /**
         * Проверяет, изменялась ли видимость страницы при загрузке.
         *
         * @returns {Boolean}
         */
        isVisibilityChanged: function() {
            return this.vsStart && (this.vsStart !== 'visible' || this.vsChanged);
        },

        /**
         * Поставить метку времени в User Timing https://www.w3.org/TR/user-timing/#dom-performance-mark
         * В браузерах, не поддерживающих window.performance.mark(), заменяется ничего не делающей заглушкой.
         *
         * @param {String} name - Код BlockStat. ID метки времени
         * @param {Number} [time] - Время. Опционально, нужно для сверки со временем, которое ставит
         *     само User Timing API, или для меток, которые ставятся "задним числом" (ttfb, ttfp etc.)
         */
        mark: perf && perf.mark ? function(name, time) {
            perf.mark(name + (time ? ': ' + time : ''));
        } : function() {
            // заглушка
        },

        /**
         * Получить значение настройки.
         *
         * @param {String} settingName
         * @returns {*|String}
         */
        getSetting: function(settingName) {
            var setting = this._settings[settingName];

            return setting === null ? null : setting || '';
        },

        /**
         * Подписаться на срабатывание временной метки
         *
         * @param {String} counterId
         * @param {Function} cb
         */
        on: function(counterId, cb) {
            if (typeof cb !== 'function') return;

            (this._markListeners[counterId] = this._markListeners[counterId] || []).push(cb);
        },

        /**
         * Отправить данные о трафике
         */
        sendTrafficData: function() {},

        /**
         * Отправить текущее значение CLS и начать считать заново
         */
        finalizeLayoutShiftScore: function() {},

        /**
         * Отправить текущее значение LCP и начать искать заново
         */
        finalizeLargestContentfulPaint: function() {}
    };

    // Отслеживаем изменения видимости вкладки
    function onVisibilityChange() {
        Ya.Rum.vsChanged = true;
        document.removeEventListener('visibilitychange', onVisibilityChange);
    }

    if (document.addEventListener) {
        document.addEventListener('visibilitychange', onVisibilityChange);
    }
})(window);
