const remark = require('remark');
const { remark: remarkWoofmd } = require('@yandex-int/remark-woofmd');
const toBemjson = require('mdast-util-to-bemjson');
const remarkDisableTokenizers = require('remark-disable-tokenizers');

const handlers = require('./src/handlers');
const anchors = require('./src/anchors');
const hrefSanitizer = require('./src/sanitizer');

const parseWikiMd = (input, settings = {}) => {
    const markdownSettings = settings.remark && settings.remark.markdown || {};
    const woofmdSettings = settings.remark && settings.remark.woofmd || {};

    if (settings.tracker_url) {
        woofmdSettings.tracker = {
            url: settings.tracker_url,
        };

        // TODO: вынести это в общий конфиг
        // тикеты на джиру внутри являются ссылками на трекер
        if (settings.tracker_url === 'https://st.yandex-team.ru') {
            woofmdSettings.tracker.aliases = 'https://jira.yandex-team.ru';
        }
    }

    const processor = remark()
        .data('settings', markdownSettings)
        .use(remarkDisableTokenizers, {
            block: [
                'indentedCode',
            ],
        })
        .use(remarkWoofmd, woofmdSettings)
        .use(anchors)
        .use(hrefSanitizer);

    const parsed = processor.parse(input);
    processor.run(parsed);

    return parsed;
};

const formatMdAst = async(ast, settings = {}) => {
    const promises = [];
    const result = toBemjson(ast, {
        handlers: handlers({
            // Этот метод надо пробрасывать через опции.
            // На клиенте и на сервере имплементация будет отличаться
            async fetchAction() {
                throw new Error('Not implemented');
            },
            ...settings,
            promises,
            parseWikiMd,
            formatMdAst,
        }),
    });

    // Хэндлеры могут пушить в этот список асинхронные задачи,
    // в которых могут мутировать конечный результат
    await Promise.all(promises);

    return result;
};

function deprecate(fn, msg) {
    return (...args) => {
        console.warn(msg);
        return fn(...args);
    };
}

const defaultExport = deprecate(
    (source, settings = {}) => {
        const mdAst = parseWikiMd(source, settings);

        return formatMdAst(mdAst, settings);
    },
    '[woofmd-to-bemjson] default export is deprecated, use parseWikiMd()+formatMdAst() instead'
);

defaultExport.parseWikiMd = parseWikiMd;
defaultExport.formatMdAst = formatMdAst;

module.exports = defaultExport;

// deprecated
defaultExport.rawToAst = deprecate(parseWikiMd,
    '[woofmd-to-bemjson] method .rawToAst() is deprecated, use parseWikiMd() instead');
defaultExport.astToBemjson = deprecate(formatMdAst,
    '[woofmd-to-bemjson] method .astToBemjson() is deprecated, use formatMdAst() instead');
