const { describe, it } = require('mocha');
const { expect } = require('chai');

const { parseWikiMd, formatMdAst } = require('../index');

function toBemjson(source, settings) {
    const mdAst = parseWikiMd(source, settings);
    return formatMdAst(mdAst, settings);
}

describe('sanitizer', () => {
    it('should sanitize ((url desc))', async() => {
        const res = await toBemjson('((javascript://any_xss xss))');

        expect(res).to.deep.equal({
            anchors: {},
            block: 'wiki-doc',
            content: {
                block: 'wiki-p',
                content: {
                    block: 'wiki-ref',
                    content: {
                        block: 'wiki-txt',
                        'wiki-attrs': {
                            txt: 'xss',
                        },
                    },
                    url: '#sanitized',
                },
            },
            mods: {
                magiclinks: 'yes',
            },
            toc: [],
            'wiki-attrs': {},
        });
    });

    it('should sanitize [xss](javascript://any_xss)', async() => {
        const res = await toBemjson('[xss](javascript://any_xss)');

        expect(res).to.deep.equal({
            anchors: {},
            block: 'wiki-doc',
            content: {
                block: 'wiki-p',
                content: {
                    block: 'wiki-ref',
                    content: {
                        block: 'wiki-txt',
                        'wiki-attrs': {
                            txt: 'xss',
                        },
                    },
                    ref: undefined,
                    title: null,
                    url: '#sanitized',
                },
            },
            mods: {
                magiclinks: 'yes',
            },
            toc: [],
            'wiki-attrs': {},
        });
    });

    it('should not sanitize [valid](http://any_xss)', async() => {
        const res = await toBemjson('[valid](http://any_xss)');

        expect(res).to.deep.equal({
            anchors: {},
            block: 'wiki-doc',
            content: {
                block: 'wiki-p',
                content: {
                    block: 'wiki-ref',
                    content: {
                        block: 'wiki-txt',
                        'wiki-attrs': {
                            txt: 'valid',
                        },
                    },
                    ref: undefined,
                    title: null,
                    url: 'http://any_xss',
                },
            },
            mods: {
                magiclinks: 'yes',
            },
            toc: [],
            'wiki-attrs': {},
        });
    });

    it('should not sanitize ((http://url desc))', async() => {
        const res = await toBemjson('((http://any_xss valid))');

        expect(res).to.deep.equal({
            anchors: {},
            block: 'wiki-doc',
            content: {
                block: 'wiki-p',
                content: {
                    block: 'wiki-ref',
                    content: {
                        block: 'wiki-txt',
                        'wiki-attrs': {
                            txt: 'valid',
                        },
                    },
                    url: 'http://any_xss',
                },
            },
            mods: {
                magiclinks: 'yes',
            },
            toc: [],
            'wiki-attrs': {},
        });
    });
});
