/**
 * Синхронный LRU кэш.
 *
 * При инстанцировании необходимо указать размер кэша,
 * самые непопулярные элементы будут выкидываться при установке новых
 * при достижении максимальгого размера
 */

// Удаляет звено из двусвязного списка
function drop(self, link) {
    if (!link) {
        return link;
    }

    if (link.prev) {
        link.prev.next = link.next;
    } else {
        self._tail = link.next;
    }

    if (link.next) {
        link.next.prev = link.prev;
    } else {
        self._head = link.prev;
    }

    link.prev = null;
    link.next = null;

    self._leng -= 1;

    return link;
}

// Добавялет звено в двусвязный список
function push(self, link) {
    if (self._head) {
        self._head.next = link;
    }

    link.prev = self._head;
    link.next = null;
    self._head = link;

    if (!self._tail) {
        self._tail = link;
    }

    self._leng += 1;
}

// Обрезает кэш до нужного размера
function crop(self, size) {
    while (self._leng > size) {
        const link = self._tail;

        if (drop(self, link)) {
            delete self._vals[link.name];

            continue;
        }

        return false;
    }

    return true;
}

function Link(name, data, prev, next) {
    this.name = name;
    this.data = data;
    this.prev = prev;
    this.next = next;
}

/**
 * @constructor LRUDict
 * @param {Number} [size]
 * */
function LruCache(size) {
    this._head = null;
    this._tail = null;
    this._size = Math.max(size, 0);
    this._leng = 0;
    this._vals = Object.create(null);
}

/**
 * Устанавливает значение в кэш
 *
 * @param {String} key
 * @param {*} val
 *
 * @returns {LruCache}
 * */
LruCache.prototype.set = function(key, val) {
    drop(this, this._vals[key]);

    if (crop(this, this._size - 1)) {
        const link = new Link(key, val, null, null);
        push(this, link);
        this._vals[key] = link;
    }

    return this;
};

/**
 * Достает из кэша значение
 *
 * @param {String} key
 *
 * @returns {*}
 * */
LruCache.prototype.get = function(key) {
    const link = drop(this, this._vals[key]);

    if (link) {
        push(this, link);

        return link.data;
    }

    return link;
};

exports.LruCache = LruCache;
