# I18N

Интерфейс для локализации текста с динамическими параметрами и плюрализацией.

## Создание словарей переводов

Файлы с переводами лежат рядом с кодом, к которому они логически относятся.

```
src/features/FeatureName
├── FeatureName.i18n — файлы переводов
│   ├── ru.ts — словарь для русского языка
│   ├── en.ts — словарь для английского языка
│   └── index.ts — словарь используемых языков
```

Файл словаря — модуль, в котором лежит кейсет для одного языка с парами `{ ключ, перевод }`:

``` ts
// FeatureName/FeatureName.i18n/en.ts
export const en = {
    Пока: 'Bye',
    Привет: 'Example',
};
```

Все языки должны быть зарегистрированы в словаре. Этим словарем оперирует сам модуль I18N.
```ts
// FeatureName/FeatureName.i18n/index.ts
export * from './ru';
export * from './en';
```


## Использование

``` ts
// FeatureName/FeatureName.ts
import i18n from '@yandex-int/i18n';

import * as keyset from './FeatureName.i18n';

const exampleI18N = i18n(keyset);
exampleI18N('Привет');
```

### Параметризация

Параметры объявляются в синтаксисе схожем с параметрами для [template strings](https://developer.mozilla.org/ru/docs/Web/JavaScript/Reference/template_strings).

``` ts
// FeatureName/FeatureName.i18n/en.ts
export const en = {
    'м. {metro}': 'subway {metro}',
}
```

``` ts
// FeatureName/FeatureName.ts
import i18n from '@yandex-int/i18n';

import * as keyset from './FeatureName.i18n';

const exampleI18N = i18n(keyset);

exampleI18N('м. {metro}', {
    metro: 'Парк Культуры',
});
```

### Плюрализация

Для выражения плюрализация существует специальный параметр `count`. Который соотносится с вариантами написания ключа через вложенные параметры: `many`, `some`, `one`, `none`.

``` ts
// FeatureName/FeatureName.i18n/en.ts
export const en = {
    '{count} оценка': {
        many: '{count} оценок',
        none: 'нет оценок',
        one: '{count} оценка',
        some: '{count} оценки',
    }
}
```

```ts
// FeatureName/FeatureName.ts
import i18n from '@yandex-int/i18n';

import * as keyset from './FeatureName.i18n';

const exampleI18N = i18n(keyset);

exampleI18N('{count} оценка', {
    count: 2,
});
```

### Интеграция с компонентами

``` ts
// FeatureName/FeatureName.i18n/en.ts
export const en = {
    'OOO «{link}»': 'LLC {link}',
    'ЯНДЕКС': 'YANDEX',
}
```

``` tsx
// FeatureName/FeatureName.tsx
import i18n, { i18nRaw } from '@yandex-int/i18n';

import * as keyset from './FeatureName.i18n';

const exampleI18NRaw = i18nRaw(keyset);
const exampleI18N = i18n(keyset);

exampleI18NRaw('OOO «{link}»', {
    link: <Link url="#">{exampleI18N('ЯНДЕКС')}</Link>
});
```

## Синхронизация с Танкером

Для синхронизаций с Танкером используйте набор хелперов для `tanker-kit` из пакета `@yandex-int/tanker-helpers`
