"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.normalizeHttpRequestParams = void 0;
var getRequestOptionsByUrl_1 = require("../../../utils/getRequestOptionsByUrl");
var getUrlByRequestOptions_1 = require("../../../utils/getUrlByRequestOptions");
var isObject_1 = require("../../../utils/isObject");
var debug = require('debug')('http normalizeHttpRequestParams');
function resolveRequestOptions(args, url) {
    // Calling `fetch` provides only URL to ClientRequest,
    // without RequestOptions or callback.
    if (['function', 'undefined'].includes(typeof args[1])) {
        return getRequestOptionsByUrl_1.getRequestOptionsByUrl(url);
    }
    return args[1];
}
function resolveCallback(args) {
    return typeof args[1] === 'function' ? args[1] : args[2];
}
/**
 * Normalizes parameters given to a `http.request` call
 * so it always has a `URL` and `RequestOptions`.
 */
function normalizeHttpRequestParams() {
    var args = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        args[_i] = arguments[_i];
    }
    var url;
    var options;
    var callback;
    debug('arguments', args);
    // Convert a url string into a URL instance
    // and derive request options from it.
    if (typeof args[0] === 'string') {
        debug('given a location string:', args[0]);
        url = new URL(args[0]);
        debug('created a URL:', url);
        options = resolveRequestOptions(args, url);
        debug('created request options:', options);
        callback = resolveCallback(args);
    }
    // Handle a given URL instance as-is
    // and derive request options from it.
    else if ('origin' in args[0]) {
        url = args[0];
        debug('given a URL:', url);
        options = resolveRequestOptions(args, url);
        debug('created request options', options);
        callback = resolveCallback(args);
    }
    // Handle a legacy URL instance and re-normalize from either a RequestOptions object
    // or a WHATWG URL.
    else if ('hash' in args[0] && !('method' in args[0])) {
        var legacyUrl = args[0];
        if (legacyUrl.hostname === null) {
            // We are dealing with a relative url, so use the path as an "option" and
            // merge in any existing options, giving priority to exising options -- i.e. a path in any
            // existing options will take precedence over the one contained in the url. This is consistent
            // with the behaviour in ClientRequest.
            // https://github.com/nodejs/node/blob/d84f1312915fe45fe0febe888db692c74894c382/lib/_http_client.js#L122
            debug('given a relative legacy URL:', legacyUrl);
            return isObject_1.isObject(args[1])
                ? normalizeHttpRequestParams(__assign({ path: legacyUrl.path }, args[1]), args[2])
                : normalizeHttpRequestParams({ path: legacyUrl.path }, args[1]);
        }
        debug('given an absolute legacy url:', legacyUrl);
        // We are dealing with an absolute URL, so convert to WHATWG and try again.
        var resolvedUrl = new URL(legacyUrl.href);
        return args[1] === undefined
            ? normalizeHttpRequestParams(resolvedUrl)
            : typeof args[1] === 'function'
                ? normalizeHttpRequestParams(resolvedUrl, args[1])
                : normalizeHttpRequestParams(resolvedUrl, args[1], args[2]);
    }
    // Handle a given RequestOptions object as-is
    // and derive the URL instance from it.
    else if (isObject_1.isObject(args[0])) {
        options = args[0];
        debug('given request options:', options);
        url = getUrlByRequestOptions_1.getUrlByRequestOptions(options);
        debug('created a URL:', url);
        callback = resolveCallback(args);
    }
    else {
        throw new Error("Failed to construct ClientRequest with these parameters: " + args);
    }
    // Enforce protocol on `RequestOptions` so when `ClientRequest` compares
    // the agent protocol to the request options protocol they match.
    // @see https://github.com/nodejs/node/blob/d84f1312915fe45fe0febe888db692c74894c382/lib/_http_client.js#L142-L145
    // This prevents `Protocol "http:" not supported. Expected "https:"` exception for `https.request` calls.
    options.protocol = options.protocol || url.protocol;
    debug('resolved URL:', url);
    debug('resolved options:', options);
    return [url, options, callback];
}
exports.normalizeHttpRequestParams = normalizeHttpRequestParams;
//# sourceMappingURL=normalizeHttpRequestParams.js.map